/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

Diagnostics and state aggregation

Copyright 2025 Tomas Krajnik 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




#include <ros/ros.h>
#include <stdio.h>
#include <rr_msgs/BeePositionArray.h>
#include <image_transport/image_transport.h>
#include <tf/tf.h>
#include <std_msgs/Int32MultiArray.h>
#include <std_srvs/SetBool.h>
#include <string.h>
#include <string>
#include <dynamic_reconfigure/server.h>
#include <rr_msgs/XYMonitorConfig.h>
#include <rr_msgs/SetCaptureInterval.h>
#include <geometry_msgs/PoseStamped.h>
#include <rr_msgs/Tweet.h>
#include <std_msgs/String.h>
#include <rr_msgs/QueenDetectionStatistics.h>
#include <time.h>
#include <stdarg.h>

typedef enum
{
	SH_OK = 0,
	SH_HW_GENERAL,
	SH_HW_CAMERAS,
	SH_HW_JETSONS,
	SH_SW_QUEEN_DETECTION,
	SH_SW_COURT
}ESystemComponent;

typedef struct
{
	ESystemComponent component;
	ros::Time lastReport;
	int numFaults;
	bool reported;
}SSystemHealth;

const char *systemHealthString[] = 
{
	"OK",
	"general hardware",
	"cameras",
	"jetsons",
	"queen detection",
	"court detection"
};

typedef enum
{
	SIDE_0 = 0,
	SIDE_1, 
	SIDE_NUMBER,
	SIDE_UNKNOWN
}ESide;

typedef enum
{
	POWER_OFF = 0,
	POWER_BOOTING = 1,
	POWER_UP = 2,
	POWER_DISABLED = 3,
	POWER_NUMBER,
	POWER_UNKNOWN
}ESidePower;


const char *sides[] = {"xy_0","xy_1","XXXXX","XXXXX"};
char topicName[1000];

/*used to switch on and off the jetsons*/
ros::ServiceClient powerClient;
ESidePower powerStatus;
int restartCount;
int totalRestartCount;
int bootupTime = 0;
int trackDuration = 0;
int sideSwaps = 0;
ros::Time powerTime;
bool allowCameraRestart = false;            //prevent node restarts  
bool allowPCRestart = false;      //prevent power shutdowns

/*used to switch between combs*/
ros::Subscriber queenDetection[SIDE_NUMBER];
ros::Publisher queenPub;
image_transport::Subscriber mainImageSub[SIDE_NUMBER];
ros::Subscriber motorSub[SIDE_NUMBER];
ros::Subscriber xyStateSub[SIDE_NUMBER];
ros::Subscriber queenStatSub[SIDE_NUMBER];
ros::Subscriber detectionStatSub[SIDE_NUMBER];
image_transport::Publisher mainImagePub;

static std::string stateDescription[SIDE_NUMBER];
static int stateDuration[SIDE_NUMBER];
int lastSide = -1;

int numImages = 0;
ros::Time lastImageReceived[SIDE_NUMBER];
bool lastImageValid[SIDE_NUMBER];

ros::Time lastMotorReceived[SIDE_NUMBER];
bool lastMotorValid[SIDE_NUMBER];
int numMotorMessages = 0;

ros::Time lastStatReceived[SIDE_NUMBER];
bool lastStatValid[SIDE_NUMBER];
int numStatMessages = 0;

int lastQueenSight[SIDE_NUMBER];
float queenTrackRatio[SIDE_NUMBER];

ros::Timer timer; 
int firstBoot;

/*ros and image transport handles*/
image_transport::ImageTransport *it;
ros::NodeHandle *nh;

/*reporting status*/
ros::Publisher twitterPub;
ros::Publisher summaryPub;
ros::Publisher numSummaryPub;
SSystemHealth systemHealth;
char str[10000];
char toSend[10000]; 
char discord[11000]; 
int myHive = -1;
int reportNumber;

const char rebootKey[] = "DISCORD WEBHOOK KEYS GO HERE";
const char warnKey[] = "DISCORD WEBHOOK KEYS GO HERE";
const char *keys[] = {"DISCORD WEBHOOK KEYS GO HERE",
                "DISCORD WEBHOOK KEYS GO HERE",
                "DISCORD WEBHOOK KEYS GO HERE",
                "DISCORD WEBHOOK KEYS GO HERE"};

char onlineMsg[] = "Diagnostics systems online."; 

void info(char *what) 
{
    ROS_INFO("%s",what);
    std_msgs::String ret;
    sprintf(toSend,"HIVE %i Report: %s",myHive,what);
    ret.data = toSend;
    summaryPub.publish(ret);
    sprintf(discord,"curl -v -H \"Authorization: Bot TOKEN\" -H \"User-Agent: GrazMasterPCBot http://roboroyale.eu, v0.1\" -H \"Content-Type: application/json\" -X POST -d \'{\"content\":\"%s\"}\' https://discord.com/api/webhooks/%s >/dev/null 2>/dev/null &",toSend,keys[myHive]);
    if (reportNumber++%6 == 0) system(discord);
}

void warn(char *what) 
{
    info(what);
    ROS_WARN("%s",what);
    std_msgs::String ret;
    sprintf(toSend,"HIVE %i WARNING: %s",myHive,what);
    ret.data = toSend;
    summaryPub.publish(ret);
    sprintf(discord,"curl -v -H \"Authorization: Bot TOKEN\" -H \"User-Agent: GrazMasterPCBot http://roboroyale.eu, v0.1\" -H \"Content-Type: application/json\" -X POST -d \'{\"content\":\"%s\"}\' https://discord.com/api/webhooks/%s >/dev/null 2>/dev/null &",toSend,warnKey);
    system(discord);
}

void rebootWarn(char *what) 
{
    info(what);
    ROS_WARN("%s",what);
    std_msgs::String ret;
    sprintf(toSend,"HIVE %i RESTART: %s",myHive,what);
    ret.data = toSend;
    summaryPub.publish(ret);
    sprintf(discord,"curl -v -H \"Authorization: Bot TOKEN\" -H \"User-Agent: GrazMasterPCBot http://roboroyale.eu, v0.1\" -H \"Content-Type: application/json\" -X POST -d \'{\"content\":\"%s\"}\' https://discord.com/api/webhooks/%s >/dev/null 2>/dev/null &",toSend,rebootKey);
    system(discord);
}

void initPCPower()
{
	std_srvs::SetBool srv;
	if (powerStatus == POWER_DISABLED  ) srv.request.data = false; else srv.request.data = true;

	if (powerClient.call(srv))
	{
		sprintf(str,"Switching computer power to %i - response is %i (1 = fine)",srv.request.data,srv.response.success);
        info(str);
	}else{
		sprintf(str,"Setting power of PC failed.");
		warn(str);
	} 

}

void restartPC()
{
    if (powerStatus == POWER_DISABLED) return;
	if (powerStatus == POWER_UP)
	{
		powerTime = ros::Time::now();
		powerStatus = POWER_OFF;
		restartCount++;
		totalRestartCount++;
        int c0 = -(lastImageReceived[0] - ros::Time::now()).sec;
        int c1 = -(lastImageReceived[1] - ros::Time::now()).sec;
        int m0 = -(lastMotorReceived[0] - ros::Time::now()).sec;
        int m1 = -(lastMotorReceived[1] - ros::Time::now()).sec;
        sprintf(str,"%i restart(s) in a row, %i total restarts. Cameras active: %i %i . Last image sent %i s %i s ago. Motors active: %i %i . Last pos sent %i s %i s ago.",restartCount,totalRestartCount,lastImageValid[0],lastImageValid[1],c0,c1,lastMotorValid[0],lastMotorValid[1],m0,m1);
        rebootWarn(str);
	}
	ros::Duration delay = ros::Time::now() - powerTime;
    if (delay.sec > 10 && powerStatus == POWER_OFF){
        powerStatus = POWER_BOOTING;
        lastImageReceived[0] =  lastMotorReceived[0]= ros::Time::now(); 
        lastImageReceived[1] =  lastMotorReceived[1]= ros::Time::now();
        lastImageValid[1] =  lastImageValid[0]= false;
    }

	if (delay.sec > restartCount*300 && powerStatus == POWER_BOOTING)
	{ 
		powerTime = ros::Time::now();
		powerStatus = POWER_OFF;
        lastImageValid[1] =  lastImageValid[0]= false;
		restartCount++;
		totalRestartCount++;
	}

	std_srvs::SetBool srv;
	if (powerStatus == POWER_OFF  ) srv.request.data = false; else srv.request.data = true;

	if (powerClient.call(srv))
	{
		if (powerStatus == POWER_BOOTING) {sprintf(str,"PC is booting for %i seconds, setting its power to %i - response is %i (1 = fine)",delay.sec,srv.request.data,srv.response.success);warn(str);}
		if (powerStatus == POWER_OFF)     {sprintf(str,"PC is off, setting its power to %i - response is %i (1 = fine)",srv.request.data,srv.response.success);warn(str);}
		if (powerStatus == POWER_UP)      {sprintf(str,"PC is up, setting its power to %i - response is %i (1 = fine)",srv.request.data,srv.response.success);warn(str);}
	}else{
		sprintf(str,"Setting power of PC failed.");
        warn(str);
	} 
}

void tweet(std::string message)
{
	rr_msgs::Tweet tweet;
	tweet.text.data = "This is not a test: " + message;
	//twitterPub.publish(tweet); 
	systemHealth.reported = false;
}


/*used to create subscribers for comb time lapse*/
void timerCallback(const ros::TimerEvent&)
{
    bool ok = true;
    ros::Duration delay;
    if (powerStatus != POWER_DISABLED){
        for (int i = 0;i<SIDE_NUMBER;i++)
        {
            delay = (ros::Time::now() - lastImageReceived[i]) + (ros::Time::now() - lastMotorReceived[i]);
            ROS_DEBUG("Camera + motor %s delay %i s",sides[i],delay.sec);
            if (delay.sec > 10){
                ok = false;
                if (delay.sec > 300){
                    systemHealth.component = SH_HW_CAMERAS;
                    sprintf(str,"Camera %s was idle for more than 5 mins after %i restart(s).",mainImageSub[i].getTopic().c_str(),restartCount);
                    warn(str);
                }
                if (delay.sec > (firstBoot*10+5)*10){
                    sprintf(str,"Side %i devices were idle for %i seconds!",i,delay.sec);
                    warn(str);
                    if (allowPCRestart) restartPC();
                }else{
                    if ((ros::Time::now() - lastImageReceived[i]).sec > 10 && lastImageValid[i]){
                        sprintf(str,"Subscriber %s did not obtain images during the last %i seconds. Camera problems!",mainImageSub[i].getTopic().c_str(),delay.sec);
                        warn(str);
                        if (allowCameraRestart){
                            char call[1000];
                            sprintf(call,"rosnode kill /hive_%i/xy_%i/camera&",myHive,i);
                            system(call);
                        }
                    }
                    if ((ros::Time::now() - lastMotorReceived[i]).sec > 10){
                        sprintf(str,"Subscriber %s did not obtain positions during the last %i seconds. Motor problems!",motorSub[i].getTopic().c_str(),delay.sec);
                        warn(str);
                    }
                }
            }else{
                if (lastImageValid[i]){
                    powerStatus = POWER_UP;
                    restartCount = 0;
                    firstBoot = 0;
                }
            }
            if (lastImageValid[i] == false) ok = false;
        }
    }
    if (ok){
        time_t seconds(lastImageReceived[0].sec); 
        tm *p = localtime(&seconds);
        int sight = fmax(lastQueenSight[0],lastQueenSight[1]);
        char lastSight[100];
        float ratio =  fmin(queenTrackRatio[0]+queenTrackRatio[1],1.0);
        sprintf(lastSight,"Queen not seen yet, %is ago.",(ros::Time::now()).sec - bootupTime);
        if (sight > 0) sprintf(lastSight,"Queen seen %is ago, tracked for %is (%.1f%%) with %i side swaps.",(ros::Time::now()).sec - sight,trackDuration/20,ratio*100,sideSwaps);

        int runTime = ros::Time::now().sec-bootupTime;
        sprintf(str,"%s Received %i image, %i motor and %i stat messages during the last 10 seconds at %02i-%02i-%02i %02i:%02i:%02i. Message delay is %is. %i restarts during the last %i days and %02i:%02i hours. Side 0 has been in state %s for %i seconds. Side 1 has been in state %s for %i seconds.",lastSight,numImages,numMotorMessages,numStatMessages,p->tm_year-100,p->tm_mon,p->tm_mday,p->tm_hour,p->tm_min,p->tm_sec,delay.sec,totalRestartCount,runTime/86400,(runTime%86400)/3600,(runTime%3600)/60,stateDescription[0].c_str(),stateDuration[0]/20,stateDescription[1].c_str(),stateDuration[1]/20);

        std_msgs::Int32MultiArray msg;
        msg.data.push_back(myHive); 
        msg.data.push_back(ratio*1000); 
        msg.data.push_back(trackDuration/20);
        msg.data.push_back(sideSwaps);
        if (sight > 0) msg.data.push_back((ros::Time::now()).sec - sight); else msg.data.push_back((ros::Time::now()).sec - bootupTime);
        numSummaryPub.publish(msg);
        info(str);
        numImages=numMotorMessages=numStatMessages=0;
    }
}

void healthTimerCallback(const ros::TimerEvent&)
{

}

/*used to get pictures of comb with the honeybee queen*/
void motorCallback0(const geometry_msgs::PoseStampedConstPtr& msg)
{
    lastMotorReceived[0] = ros::Time::now();
    lastMotorValid[0] = true;
    numMotorMessages++;
}

void motorCallback1(const geometry_msgs::PoseStampedConstPtr& msg)
{
    lastMotorReceived[1] = ros::Time::now();
    lastMotorValid[1] = true;
    numMotorMessages++;
}

void stateCallback0(const std_msgs::StringConstPtr& msg)
{
    int n = 0;
    if (stateDescription[n] != msg->data) stateDuration[n] = 0; else stateDuration[n]++;
    if (msg->data == "Track-queen" || stateDescription[0] == "Track-queen" || stateDescription[1] == "Track-queen") trackDuration++; else trackDuration = 0; 
    if (msg->data == "Track-queen" && lastSide == -1) lastSide = 0;
    if (msg->data == "Track-queen" && stateDescription[1-n] != "Track-queen" && trackDuration/20 > 30){
	    if (lastSide == 1) sideSwaps++;
	    lastSide = 0;
    }
    stateDescription[n] = msg->data;
}

void stateCallback1(const std_msgs::StringConstPtr& msg)
{
    int n = 1;
    if (stateDescription[n] != msg->data) stateDuration[n] = 0; else stateDuration[n]++;
    if (msg->data == "Track-queen" && lastSide == -1) lastSide = 1;
    if (msg->data == "Track-queen" && stateDescription[1-n] != "Track-queen" && trackDuration/20 > 30)
    {
	    if (lastSide == 0) sideSwaps++;
	    lastSide = 1;
    }
    stateDescription[n] = msg->data;
}



void statCallback(const rr_msgs::QueenDetectionStatisticsConstPtr& msg)
{
    int side = msg->side;
    if (side == 0 || side == 1){
        lastStatReceived[side] = ros::Time::now();
        lastStatValid[side] = true;
        lastQueenSight[side] = msg->lastQueenDetection;
        queenTrackRatio[side] = msg->detectionRatio;
    }else{
        sprintf(str,"Indicated side of queen detection stat message is %i, which is out of range.",side);
        warn(str);
    }
    numStatMessages++;
}



/*used to get pictures of comb with the honeybee queen*/
void mainImageCallback(const sensor_msgs::ImageConstPtr& msg)
{
	ROS_DEBUG("Image %s",msg->header.frame_id.c_str());
	int side = 0;
	for (side = 0;side<SIDE_NUMBER;side++){
	       	if (msg->header.frame_id.find(sides[side]) != -1) break;
	}
	if (side > -1 && side < SIDE_NUMBER)	
	{
		lastImageReceived[side] = ros::Time::now();
        lastImageValid[side] = true;
        numImages++;
		//imagePub[side].publish(msg);
	}
}

/*used for side time lapse*/
void timeLapseCallback(const sensor_msgs::ImageConstPtr& msg)
{

}


/* dynamic reconfigure of data collection rates */
void reconfig(rr_msgs::XYMonitorConfig &config, uint32_t level)
{
    allowPCRestart = config.allowPCRestarts;
    allowCameraRestart = config.allowCameraNodeRestarts;
	ROS_INFO("Camera restarts %i PC restarts %i",allowCameraRestart,allowPCRestart);
}

int main(int argc, char **argv)
{
	ros::init(argc, argv, "xy_monitor");
	nh = new ros::NodeHandle();
	it = new image_transport::ImageTransport(*nh);

	systemHealth.reported = true;
	systemHealth.lastReport = ros::Time(0);
	systemHealth.numFaults = 0;
	systemHealth.component = SH_OK;
	//lastQueenDetection = ros::Time(0); 

	lastImageReceived[0] = lastImageReceived[1] = ros::Time::now();
	lastImageValid[0] = lastImageValid[1] = false;
	lastMotorReceived[0] = lastMotorReceived[1] = ros::Time::now();
	lastMotorValid[0] = lastMotorValid[1] = false;
	lastStatReceived[0] = lastStatReceived[1] = ros::Time::now();
	lastStatValid[0] = lastStatValid[1] = false;
	lastQueenSight[0] = queenTrackRatio[0] = lastQueenSight[1] = queenTrackRatio[1] = 0;
	firstBoot = 1;

	for (int i = 0;i<10;i++){
		char hiveName[50];
		sprintf(hiveName,"hive_%i",i);
		if (std::string::npos != ros::this_node::getNamespace().find(hiveName)) myHive = i;
	}
	summaryPub = nh->advertise<std_msgs::String>("summary", 1);
	numSummaryPub = nh->advertise<std_msgs::Int32MultiArray>("numerical_summary", 1);
	/*sprintf(topicName,"%s/queen_position",combs[i]);
	  ROS_INFO("Subscribing to %s ",topicName);
	  queenDetection[i] = nh->subscribe(topicName, 1,queenDetectionCallback);*/

	//sprintf(topicName,"%s/power_plug/set_state",sides[i]);
	ROS_INFO("Connecting to services %s ","pc/power_plug/set_state");
	powerClient = nh->serviceClient<std_srvs::SetBool>("pc/power_plug/set_state");
	powerStatus = POWER_BOOTING;

	restartCount = 0;
	totalRestartCount = 0;
	bootupTime = ros::Time::now().sec;
	initPCPower();

	/* Initiate dynamic reconfiguration */
	dynamic_reconfigure::Server<rr_msgs::XYMonitorConfig> server;
	dynamic_reconfigure::Server<rr_msgs::XYMonitorConfig>::CallbackType f = boost::bind(&reconfig, _1, _2);
	server.setCallback(f);


	for (int i = 0;i<SIDE_NUMBER;i++)
	{
		sprintf(topicName,"xy_%i/camera/image_raw_status",i);
		ROS_INFO("Subscribing to %s",topicName);
		mainImageSub[i] = it->subscribe(topicName, 1,mainImageCallback);
		sprintf(topicName,"xy_%i/queen_detection_statistics",i);
		queenStatSub[i] = nh->subscribe(topicName, 1,statCallback);
	}

	sprintf(topicName,"xy_%i/xy_position",0);
	ROS_INFO("Subscribing to %s",topicName);
	motorSub[0] = nh->subscribe(topicName, 1,motorCallback0);

	sprintf(topicName,"xy_%i/xy_position",1);
	ROS_INFO("Subscribing to %s",topicName);
	motorSub[1] = nh->subscribe(topicName, 1,motorCallback1);

	sprintf(topicName,"xy_%i/state",0);
	xyStateSub[0] = nh->subscribe(topicName, 1,stateCallback0);

	sprintf(topicName,"xy_%i/state",1);
	xyStateSub[1] = nh->subscribe(topicName, 1,stateCallback1);

	//motorSub[i] = nh->subscribe(topicName, 1,motorCallback);
	timer = nh->createTimer(ros::Duration(10), timerCallback);
	/*sprintf(topicName,"image_mono");
	  fullImagePub = it->advertise(topicName, 1);
	  sprintf(topicName,"queen_position");
	  queenPub = nh->advertise<rr_msgs::BeePositionArray>(topicName, 1);
	  twitterPub = nh->advertise<rr_msgs::Tweet>("/tweet", 1);*/
    warn(onlineMsg);
	ros::spin();
	return 0;
}


