/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

XY actuator high-level control

Copyright 2025 Tomas Krajnik 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




#include <sensor_msgs/JointState.h>
#include <ros/ros.h>
#include <image_transport/image_transport.h>
#include <cv_bridge/cv_bridge.h>
#include <sensor_msgs/image_encodings.h>
#include <opencv2/opencv.hpp>
#include <stdio.h>
#include <iostream>
#include <algorithm>
#include <cmath>
#include <dynamic_reconfigure/server.h>
#include <rr_msgs/BeePositionArray.h>
#include <rr_msgs/BeePosition.h>
#include <rr_msgs/ActuatorStatus.h>
#include <rr_msgs/CropperConfig.h>
#include <rr_msgs/SetCaptureInterval.h>
#include <std_msgs/Int32.h>
#include <std_msgs/Float32MultiArray.h>
#include <geometry_msgs/Twist.h>
#include <actionlib/server/simple_action_server.h>
#include <rr_xy_operator/xy_operatorAction.h>
#include <rr_xy_operator/xy_operatorConfig.h>
#include <geometry_msgs/PoseStamped.h>
#include <whycon/MarkerArray.h>
#include <std_srvs/Trigger.h>
#include <std_msgs/String.h>
#include <rr_xy_operator/ResetPosition.h>
#include <rr_msgs/LensStatus.h>
#include <rr_msgs/QueenDetectionStatistics.h>
#include <rr_msgs/LensFocus.h>
#include <rr_msgs/LensZoom.h>
#include <rr_msgs/ScanPositionArray.h>
#include <rr_msgs/ScanPosition.h>
#include <std_srvs/Empty.h>

typedef enum
{
	IDLE,
	DOCK,
	LEAVE_DOCK,
	CALIB_ADJUST,
	CALIB_MOVE,
	SCAN_INIT,
	SCAN_ADJUST,
	SCAN_MOVE,
	SCAN_WAIT,
	TRACK_QUEEN,
	TRACK_WORKER,
	GOTO_CELL,
	SCAN_CELL,
	MIRROR,
	QUEEN_SEARCH,
	FINISHED,
	PREEMPTED,
	FAILED,
	NUMBER
}EState;

const char *stateStr[] = 
{
   "Idle",
   "Dock",
   "Leaving-dock",
   "Calibration-adjust-to-tag",
   "Calibration-move",  
   "Initialising-scan",
   "Scan-adjust-to-tag",
   "Scan-move",
   "Scan-capture",
   "Track-queen",
   "Track-worker",
   "Goto_cell",
   "Scan_cell",
   "Mirror",
   "Queen_search",
   "Finished",
   "Preempted",
   "Failed",
   "Number"
};

const char *stateToStr(EState state)
{
    return stateStr[(int)state];
}

/*namespaces*/
using namespace cv;
using namespace std;


int reportImageCountdown = 0;
/*calibration parameters and variables*/
// idx: 0 - origin idx: 1 - x-axis  idx: 2 - xy-point idx: 3 - y-axis
float tagMotorX[5];
float tagMotorY[5];
//float firstTagX = .03541198730468742;
//float firstTagY = .007745999813079834; 
float firstTagX = .01;
float firstTagY = .03; 
bool forceZoom = false;

int minQueenMarkerSize = 150;
int maxQueenMarkerSize = 3500;
int minCalibrationMarkerSize = 10000;
int maxCalibrationMarkerSize = 150000;

float tagCameraX[5] = {0.000,0.495,0.495,0.000,0.000};
float tagCameraY[5] = {0.000,0.000,0.400,0.400,0.000};
float homDirect[9] = {1.0,0.0,0.0, 0.0,1.0,0.0, 0.0,0.0,1.0};
float homInverse[9] = {1.0,0.0,0.0, 0.0,1.0,0.0, 0.0,0.0,1.0};
float homEye[9] = {1.0,0.0,0.0, 0.0,1.0,0.0, 0.0,0.0,1.0};
int scanCount = 1;
int maxScanCount = 10;

//float searchPointX[] = {0.11,0.42,0.42,0.11,0.11,0.42,0.42,0.11};
//float searchPointY[] = {0.10,0.10,0.26,0.26,0.33,0.33,0.48,0.48};
float outlineX[] = {0.08,0.45,0.45,0.08,0.08,0.45,0.45,0.08};
float outlineY[] = {0.07,0.07,0.29,0.29,0.30,0.30,0.51,0.51};
float searchPointX[] = {0.10,0.43,0.43,0.10,0.10,0.43,0.43,0.10};
float searchPointY[] = {0.09,0.09,0.295,0.295,0.295,0.295,0.50,0.50};
float sweepPointX[7];
float sweepPointY[7];
int sweepStep = 0;

/*calibration variables*/
int tagNumber = 0;
float tagMeasurementX = 0;
float tagMeasurementY = 0;
int adjustCount = 0;
int adjustCountMax = 50;
float maxForce = 0.01;


/*subscribers and publishers*/
ros::Subscriber beePositionSub;
ros::Subscriber markerSub;
ros::Subscriber xyStatusSub;
ros::Subscriber xyPositionSub;
ros::Subscriber camInfoSub;
ros::Subscriber lensStatusSub;
image_transport::Subscriber imageSub;
ros::Subscriber otherXYPositionSub;
ros::Subscriber otherXYQueenSub;
ros::Subscriber otherXYStateSub;
ros::Subscriber otherXYStatsSub;

ros::Timer queenPublishTimer;
ros::Publisher lensZoomPub; 
ros::Publisher lensFocusPub; 
ros::Publisher xyCmd; 
ros::Publisher xyCmdMirror; 
ros::Publisher cameraPositionPub; 
ros::Publisher camInfoPub;
ros::Publisher queenPositionPub;
ros::Publisher statePub;
ros::Publisher queenDetectionStatisticsPub;
image_transport::Publisher snapshotPub;
image_transport::Publisher zoomedSnapshotPub;
image_transport::Publisher cellPub;
image_transport::Publisher queenPub;
image_transport::Publisher workersPub;
image_transport::Publisher zoomedPub;
image_transport::Publisher reportPub;

ros::Timer timer; 
ros::ServiceClient resetPositionClient;

/*ros and image transport handles*/
ros::NodeHandle *nh;
image_transport::ImageTransport *it;
geometry_msgs::Twist twist;

/*currents to the motors*/
float currentX = 0;
float currentY = 0;
float currentLimit = 0;

/*state variables*/
std::string otherStateStr = "";
int otherLastQueenDetection = 0;
EState state = IDLE;
EState prevState = IDLE;
EState preWaitState = IDLE;
int zoomWait = 0;
float motorX = 0;
float motorY = 0;
float lastMotorX = 0;
float lastMotorY = 0;
float cameraPositionX = 0;
float cameraPositionY = 0;
float requiredPositionX = 0; 
float requiredPositionY = 0; 
float queenPositionX = 0; 
float queenPositionY = 0; 
float lastQueenPositionX = 0; 
float lastQueenPositionY = 0; 
float tagX = 0;
float tagY = 0;
float currentTrackLength = 0;
float totalTrackLength = 0;
int currentWalkDuration = 0;
int totalWalkDuration = 0;
int currentRestDuration = 0;
int totalRestDuration = 0;


float adjustSpeedCoef = 0.5;

/*calibration variables*/
int maxMarkerSize = 150000;
int minMarkerSize = 10000;
int minQueenSize = 500;
int maxQueenSize = 1000;
int expectedMarkerSize = 40000;
float adjustLimit = 0.001; //TODO raise this to 0.0005 
float requiredDistanceX = 0;
float requiredDistanceY = 0;
bool calibrated = false;

/*initial docking*/ 
float dockSpeed = 0.01;
bool switchLeft = false;
bool switchRight = false;
bool switchBottom = false;
bool switchTop = false;

/*scan-related variables*/
int scanningWaitFrameCount = 0;
int scanningWaitFrameLimit = 14;             //determines how many pictures to skip when observing scan target location, i.e. fps = 30/(1+scanningWaitLimit)
int scanningMoveFrameCount = 0;
int scanningMoveFrameLimit = 14;	//determines how many pictures to skip when observing scan target location, i.e. fps = 30/(1+scanningMoveFrameLimit)
rr_msgs::ScanPositionArray scanPositionArray;
int scanPositionIndex = 0;
bool scanFirstMove = false;
int scanPositionIndexZoom = 0;
int scanPositionIndexUnzoom = 0;
int scanID = 0;

int cellScanFrameCount = 0;
bool zoomedScan = false;
int scanPublishedImages = 0;
int scanRequiredImages = 1;             //set to 1 normally, 25 for deep scan

/*tracking variables*/
int trackingMisses = 0;
int trackingWaitTime = 120;      //how long to wait when not seeing the queen
int mirrorWaitTime = 60;      //how long to wait when not seeing the queen
int trackingHits = 0;
int queenSearchCountdown = 1;   //when set to 0, searching for queen

/*statistics*/
bool firstPositionRead = false;
float travelledX = 0;
float travelledY = 0;
float totalTravelledX = 0;
float totalTravelledY = 0;
float driftX = 0;
float driftY = 0;
float lastCameraPositionX = 0;
float lastCameraPositionY = 0;
bool testingDrift = false;
ros::Time queenTrackingStart;
ros::Time queenTrackingEnd;
ros::Time nodeStart;
int lastFullScanTime;
int lastZoomedScanTime;
ros::Time lastQueenDetection;
int trackingDuration = 0;
int stuckCount = 0;
int adjustStuckCount = 0;

rr_msgs::BeePositionArray markers;

/*action server variables*/
typedef actionlib::SimpleActionServer<rr_xy_operator::xy_operatorAction> Server;
Server *server;
rr_xy_operator::xy_operatorResult result;
rr_xy_operator::xy_operatorFeedback feedback;

/*camera related parameters*/
Mat intrinsic(3,3,CV_32FC1);
Mat distortion = cv::Mat::zeros(1,5, CV_32FC1);
bool distortionKnown = false;
bool imageStill = false;
std_msgs::Header header;


/*reconfiguration*/
int  queenSearchCountdownInit = 1;	//how many full scans before searching for the queen 
float  inspectedCellX = 0.13 ;	//coordinates of the inspected cell 
bool  forceInspect = false;	//manually forced inspection 
float  inspectedCellY = 0.085;	//coordinates of the inspected cell 
int  inspectedCellDuration = 60;	//time to stare at the cell
bool  inspectedCellEnable = true;	//perform cell inspection
bool  lateInspect = false;	//perform cell inspection after queen tracking (if false, inspection is performed after the calibration)
ros::Time cellInspectionStart;
ros::Time homingStart;
int zoomInVal,irisInVal,zoomOutVal,focusInVal,focusOutVal,irisOutVal;
double desiredZoom; 
double desiredFocus;
int zoomOutCalled = 0;
int zoomInCalled = 0;
int currentZoom; 
int currentFocus; 
bool cameraFocused = true; 
bool cameraZoomed = true; 
ros::ServiceClient focusHomeClient;
ros::ServiceClient zoomHomeClient;
int workersPublishInterval = 300; 
int workersWaitInterval = 3600-workersPublishInterval; 

/*determine if queen will move*/
float anchorX = 0;
float anchorY = 0;
float idleTime = 0;
float idleMirrorTime = 1000;

/*borders*/
float borderTop     = 0.03;
float borderBottom  = 0.01;
float borderLeft    = 0.00;
float borderRight   = 0.00;

/*manipulator side data*/
char otherXYPositionName[100];
char otherXYQueenName[100];
char otherXYStateName[100];
char otherXYStatsName[100];
char mySide = 0;
char myHive = 0;

/*randomized publishing timer*/
ros::Time publishWorkersStart;



bool configureDetection(int minSize,int maxSize,int number)
{
    dynamic_reconfigure::ReconfigureRequest srv_req;
    dynamic_reconfigure::ReconfigureResponse srv_resp;
    dynamic_reconfigure::IntParameter param;
    dynamic_reconfigure::Config conf;

    param.name = "num_markers";
    param.value = number;
    conf.ints.push_back(param);
    param.name = "min_size";
    param.value = minSize;
    conf.ints.push_back(param);
    param.name = "max_size";
    param.value = maxSize;
    conf.ints.push_back(param);
    srv_req.config = conf;
    bool result = ros::service::call("whycon/set_parameters", srv_req, srv_resp);
    if ( result == false) ROS_WARN("Whycon reconfigured to detect markers of size %d to %d.",minSize,maxSize);
    return result;
}

void zoomIn()
{
    if (zoomInCalled < 10){
        desiredZoom =  zoomInVal;
        desiredFocus = focusInVal;
        rr_msgs::LensZoom zoom;
        rr_msgs::LensFocus focus;
        zoom.zoom = desiredZoom;
        focus.focus = desiredFocus;
        lensZoomPub.publish(zoom);
        lensFocusPub.publish(focus);
        ROS_INFO("Zoom in called");
        zoomInCalled++;
        zoomOutCalled = 0;
    }
}

void zoomOut()
{
    if (zoomOutCalled < 10){
        desiredZoom =  zoomOutVal;
        desiredFocus = focusOutVal;
        rr_msgs::LensZoom zoom;
        rr_msgs::LensFocus focus;
        zoom.zoom = desiredZoom;
        focus.focus = desiredFocus;
        lensZoomPub.publish(zoom);
        lensFocusPub.publish(focus);
        ROS_INFO("Zoom out called %i %i",zoom.zoom,focus.focus);
        zoomOutCalled++;
        zoomInCalled = 0;
    }
}


void applyHomography(float *matrix,float *cur_x, float *cur_y)
{
	// homography transformation from actuator coords into the world coords or vice versa
	float new_x = matrix[0] * cur_x[0] + matrix[1] * cur_y[0] + matrix[2];
	float new_y = matrix[3] * cur_x[0] + matrix[4] * cur_y[0] + matrix[5];
	float scale = matrix[6] * cur_x[0] + matrix[7] * cur_y[0] + matrix[8];

	cur_x[0] = new_x / scale;
	cur_y[0] = new_y / scale;
}


void printHomoState()
{
    for (int i = 0;i<4;i++){
        printf("Homo: %.4f %.4f %.4f %.4f\n",tagMotorX[i], tagMotorY[i], tagCameraX[i], tagCameraY[i]);
    }
    for (int i = 0;i<4;i++) printf("%.4f, ",tagMotorX[i]);
    printf("\n");
    for (int i = 0;i<4;i++) printf("%.4f, ",tagMotorY[i]);
    printf("\n");
    for (int i = 0;i<4;i++) printf("%.4f, ",tagCameraX[i]);
    printf("\n");
    for (int i = 0;i<4;i++) printf("%.4f, ",tagCameraY[i]);
    printf("\n");
}

bool positionReached(float tolerance = 0.0001)
{
	float dx = requiredPositionX -  motorX;
	float dy = requiredPositionY -  motorY;
	ROS_DEBUG("REQUIRED %f %f, %f %f, %f %f, %f %f",cameraPositionX,cameraPositionY,requiredPositionX,requiredPositionY,motorX,motorY,dx,dy);
	return sqrt(dx*dx+dy*dy) < tolerance;
}

void gotoAbsolute(float ix,float iy,float maxSpeed = 0.01,bool overLimits = false)
{
	sensor_msgs::JointState requested;
	float x = ix;
	float y = iy;
	if (overLimits == false){
		if (x > tagCameraX[2]+firstTagX+borderRight) x = tagCameraX[2]+borderRight+firstTagX;
		if (y > tagCameraY[2]+borderTop+firstTagY) y = tagCameraY[2]+borderTop+firstTagY;
		if (x < tagCameraX[0]-borderLeft) x = tagCameraX[0]-borderLeft;
		if (y < tagCameraY[0]-borderBottom) y = tagCameraY[0]-borderBottom;
	}
	if (calibrated) applyHomography(homInverse,&x, &y);
	requiredPositionX = x; 
	requiredPositionY = y;
	ROS_DEBUG("Moving to %f %f, which is %f %f in motor coords.",ix,iy,x,y);
	requested.position.push_back(x);
	requested.position.push_back(y);
	requested.velocity.push_back(maxSpeed);
	requested.velocity.push_back(maxSpeed);
	requested.effort.push_back(maxForce);
	requested.effort.push_back(maxForce);
	xyCmd.publish(requested);
}

void gotoRelative(float rx,float ry,float maxSpeedX = 0.005,float maxSpeedY = 0.005,bool overLimits = false)
{
	sensor_msgs::JointState requested;
	float x = rx+cameraPositionX; 
	float y = ry+cameraPositionY;
	if (overLimits == false){
		if (x > tagCameraX[2]+borderRight) x = tagCameraX[2]+borderRight;
		if (y > tagCameraY[2]+borderTop) y = tagCameraY[2]+borderTop;
		if (x < tagCameraX[0]-borderLeft) x = tagCameraX[0]-borderLeft;
		if (y < tagCameraY[0]-borderBottom) y = tagCameraY[0]-borderBottom;
	}
	applyHomography(homInverse,&x, &y);
	requiredPositionX = x;
	requiredPositionY = y;
	requested.position.push_back(x);
	requested.position.push_back(y);
	requested.velocity.push_back(maxSpeedX);
	requested.velocity.push_back(maxSpeedY);
	requested.effort.push_back(maxForce);
	requested.effort.push_back(maxForce);
	xyCmd.publish(requested);
}


void calculateHomography(float *from_x, float *from_y, float *to_x, float *to_y)
{
	cv::Matx<float, 8, 9> mat;
	for (int i = 0; i < 4; i++) {
        ROS_INFO("Calibration points: motor: %f %f tags: %f %f",from_x[i],from_y[i],to_x[i],to_y[i]);
		mat(2 * i, 0) = -from_x[i];
		mat(2 * i, 1) = -from_y[i];
		mat(2 * i, 2) = -1;
		mat(2 * i, 3) = 0;
		mat(2 * i, 4) = 0;
		mat(2 * i, 5) = 0;
		mat(2 * i, 6) = to_x[i] * from_x[i];
		mat(2 * i, 7) = to_x[i] * from_y[i];
		mat(2 * i, 8) = to_x[i];
		mat(2 * i + 1, 0) = 0;
		mat(2 * i + 1, 1) = 0;
		mat(2 * i + 1, 2) = 0;
		mat(2 * i + 1, 3) = -from_x[i];
		mat(2 * i + 1, 4) = -from_y[i];
		mat(2 * i + 1, 5) = -1;
		mat(2 * i + 1, 6) = to_y[i] * from_x[i];
		mat(2 * i + 1, 7) = to_y[i] * from_y[i];
		mat(2 * i + 1, 8) = to_y[i];
	}

	cv::Mat w;
	cv::Mat u;
	cv::Mat vt;
	cv::SVD::compute(mat, w, u, vt, cv::SVD::FULL_UV);

	cv::Mat h_vec = vt.row(vt.rows - 1);
	cv::Mat h = cv::Mat::eye(3, 3, CV_32FC1);;
	cv::Mat hi = cv::Mat::eye(3, 3, CV_32FC1);;
	h_vec = h_vec / h_vec.at<float>(h_vec.rows - 1, h_vec.cols - 1);
	for (int i = 0; i < 9; i++){
	       	homDirect[i] = h_vec.at<float>(i);
		h.at<float>(i/3,i%3)=h_vec.at<float>(i);
	}
	hi = h.inv();
	for (int i = 0; i < 9; i++){
	       	homInverse[i] = hi.at<float>(i/3,i%3);
	}
	std::cout << h  << std::endl <<  std::endl;
	std::cout << hi << std::endl << std::endl;
}

void generateScanPositions(float scanStep, bool quick,float velocity = 0.01)
{
	rr_msgs::ScanPosition pos;
	pos.x =  fmax(tagCameraX[0],tagCameraX[3])+scanStep/2.0;
	pos.y =  fmax(tagCameraY[0],tagCameraY[1])+scanStep/2.0;
	pos.v =  velocity;
	pos.numPictures = 1;
	pos.continuousCapture = true;
	pos.tolerance = 0.0001;
	if (quick){
		pos.tolerance = 0.01;
		pos.numPictures = 0;
		pos.continuousCapture = true;
	}
	scanPositionArray.pos.clear();
	while (pos.y < fmin(tagCameraY[2],tagCameraY[3]))
	{
		if (quick == false || scanPositionArray.pos.size() == 0) scanPositionArray.pos.push_back(pos); 
		pos.x += scanStep;
		if (pos.x > fmin(tagCameraX[1],tagCameraX[2])){
		       	scanStep = -fabs(scanStep);
			pos.x += scanStep;
			if (quick) scanPositionArray.pos.push_back(pos); 
			pos.y += fabs(scanStep);
			if (quick && pos.y < fmin(tagCameraY[2],tagCameraY[3])) scanPositionArray.pos.push_back(pos);
		}
		if (pos.x < fmax(tagCameraX[0],tagCameraX[3])){
		       	scanStep = +fabs(scanStep);
			pos.x = scanStep/2.0;
			if (quick) scanPositionArray.pos.push_back(pos); 
			pos.y += fabs(scanStep);
			if (quick && pos.y < fmin(tagCameraY[2],tagCameraY[3])) scanPositionArray.pos.push_back(pos);
		}
	}
	if (scanPositionArray.pos.size()>0)	scanPositionArray.pos[0].continuousCapture = false;
//	for (int i = 0;i<scanPositionArray.pos.size();i++) ROS_INFO("%.3f %.3f",scanPositionArray.pos[i].x,scanPositionArray.pos[i].y);
	ROS_INFO("Generated %ld scan positions, current position is %i, scanID is %i",scanPositionArray.pos.size(),scanPositionIndex,scanID);
}


void initiateScan(bool zoomed)
{

	int dNormal =  ros::Time::now().sec - lastFullScanTime;
	int dZoomed =  ros::Time::now().sec - lastZoomedScanTime;
	state = SCAN_INIT;
	ROS_INFO("Last normal scan completed %i seconds ago", dNormal);
	ROS_INFO("Last zoomed scan completed %i seconds ago", dZoomed);
	zoomedScan = zoomed;
	queenSearchCountdown =  0;

	if (zoomedScan){
		generateScanPositions(0.025,false);
		scanPositionIndex =  scanPositionIndexZoom;
	}else{
		generateScanPositions(0.065,true,0.005);
		scanPositionIndex =  scanPositionIndexUnzoom;
	}

	configureDetection(minQueenMarkerSize,maxQueenMarkerSize,1);
}


void tagPositionCallback(const whycon::MarkerArray::ConstPtr& msg)
{
	int number = msg->markers.size();
	markers.header = msg->header;
	float vx = 0;
	float vy = 0;


	if (number > 0) 
	{
		int index = -1;
		int minDifference = 10000000;
		for (int i = 0; i < number; ++i)
		{
			if (msg->markers[i].size < maxMarkerSize && msg->markers[i].size > minMarkerSize) 
			{
				int difference = abs(expectedMarkerSize-msg->markers[i].size);
				if (minDifference > difference) {
					minDifference = difference;
					index = i;
				}
			}
		}
		//		ROS_INFO("Size: %i Count %i", msg->markers[0].size,index);

		if (index != -1) 
		{
			// updating queen position from whycon
			if (state == CALIB_ADJUST || state == SCAN_ADJUST)
			{
				adjustStuckCount = 0;
				tagX = -msg->markers[index].position.position.y;
				tagY = +msg->markers[index].position.position.z;
				//ROS_INFO("TAG: %.4f %.4f %i",tagX,tagY,adjustCount);
				if (fabs(tagX) < adjustLimit && fabs(tagY) < adjustLimit)
				{
					tagMeasurementX+=motorX;
					tagMeasurementY+=motorY;
					adjustCount++;
				}else{
					tagMeasurementX = 0;
					tagMeasurementY = 0;
					adjustCount = 0;
				}
				if (adjustCount >= adjustCountMax)
				{
					tagMotorX[tagNumber] = tagMeasurementX/adjustCount;
					tagMotorY[tagNumber] = tagMeasurementY/adjustCount;
					tagNumber++;
					adjustCount = 0;
					tagMeasurementX = 0;
					tagMeasurementY = 0;

					if (state == CALIB_ADJUST)
					{
						ROS_INFO("Adjust step %i of calibration is done",tagNumber);
						if (tagNumber == 1){
							tagMotorX[0] = motorX = firstTagX; 
							tagMotorY[0] = motorY = firstTagY; 
							//ROS_INFO("Homing step %i of calibration is done: %0.4f %0.4f Size: %i Count %i", tagNumber, initialX, initialY, msg->markers[index].size,adjustCount);
							rr_xy_operator::ResetPosition srv;
							srv.request.x = firstTagX; //TODO
							srv.request.y = firstTagY;
							if (!resetPositionClient.call(srv)){
								ROS_ERROR("Failed to call position reset service.");
							}else{
								ROS_INFO("Table offset was %.5f %5f.",srv.request.x-srv.response.xMeasure,srv.request.y-srv.response.yMeasure);
							}
						}
						state = CALIB_MOVE;
						if (tagNumber == 4) 
						{
							calculateHomography(tagMotorX, tagMotorY, tagCameraX, tagCameraY);
							configureDetection(minQueenMarkerSize,maxQueenMarkerSize,1);
							initiateScan(false);
							tagNumber = 0;
							calibrated = true;
						}else{
							configureDetection(minCalibrationMarkerSize,maxCalibrationMarkerSize,0);
						}

					}
					/*post-calibration adjustment*/
					if (state == SCAN_ADJUST)
					{
						rr_xy_operator::ResetPosition srv;
						srv.request.x = firstTagX; //TODO
						srv.request.y = firstTagY;
						if (!resetPositionClient.call(srv)){
							ROS_ERROR("Failed to call position reset service.");
						}else{
							ROS_INFO("Table offset was %.5f %5f, queen search countdown is %i.",srv.request.x-srv.response.xMeasure,srv.request.y-srv.response.yMeasure, queenSearchCountdown);
						}
						travelledX = 0;
						travelledY = 0;
						state = SCAN_MOVE;
						scanFirstMove = true;
						ROS_INFO("Scan pos %i %.3f %.3f ",scanPositionIndex,scanPositionArray.pos[scanPositionIndex].x,scanPositionArray.pos[scanPositionIndex].y);
						if (zoomedScan) zoomIn(); else zoomOut();
						configureDetection(minQueenMarkerSize,maxQueenMarkerSize,1);

						tagNumber = 0;
						
					
						if (inspectedCellEnable && lateInspect == false && zoomedScan == false) state = GOTO_CELL;	
					}

				} 
				//ROS_INFO("Whycon reports: %0.4f %0.4f Size: %i Count %i", tagX, tagY, msg->markers[index].size,adjustCount);
				gotoRelative(tagX*adjustSpeedCoef,tagY*adjustSpeedCoef,fabs(tagX)*adjustSpeedCoef,fabs(tagY)*adjustSpeedCoef,true);
			}

		}
	}
	if (prevState != state) ROS_INFO("Switched to %s",stateToStr(state));
	prevState = state;
}

void publishCameraInfo()
{
    if (distortionKnown){ 
        sensor_msgs::CameraInfo zoomCameraInfo;
        for (int i = 0;i<9;i++) zoomCameraInfo.K[i] = intrinsic.at<float>(i/3,i%3);
        for (int i = 0;i<5;i++) zoomCameraInfo.D.push_back(distortion.at<float>(i));
        camInfoPub.publish(zoomCameraInfo);
    }
}


void imageCallback(const sensor_msgs::ImageConstPtr& msg)
{
	/*if (distortionKnown){
	  header = msg->header;
	  cv::Mat a = cv::Mat(msg->height, msg->width, CV_8UC1, (void*)msg->data.data());
	  cv::Mat b(msg->height, msg->width, CV_8UC1);;
	  undistort(a,b, intrinsic,distortion);
	  }*/
	if (forceZoom)
	{
		//gotoRelative(0,0,0);
		if (zoomWait--<0){
			zoomOut();
			forceZoom = false;
		}
	}
	if (state == SCAN_MOVE && scanPositionArray.pos[scanPositionIndex].continuousCapture)
	{
		if (scanningMoveFrameCount++ >= scanningMoveFrameLimit){
			if (scanFirstMove == false){
				if (zoomedScan) zoomedSnapshotPub.publish(msg); else snapshotPub.publish(msg);
			}
			scanningMoveFrameCount = 0;
		}
	}

	if (state == SCAN_WAIT)
	{
		ROS_DEBUG("Scanning %i",scanningWaitFrameCount);
		scanFirstMove = false;
		if (scanningWaitFrameCount++ >= scanningWaitFrameLimit)
		{
			if (zoomedScan) zoomedSnapshotPub.publish(msg); else snapshotPub.publish(msg);
			publishCameraInfo();
			scanningWaitFrameCount = 0;
			scanPublishedImages++;
			if (scanPublishedImages >= scanRequiredImages) 
			{
				scanPositionIndex++;
				scanPublishedImages = 0;
				state = SCAN_MOVE;	
					
				if (scanPositionIndex >= scanPositionArray.pos.size())
				{
					scanPositionIndex = 0;
					scanID++;
					state = SCAN_INIT;
					if (zoomedScan){
						lastZoomedScanTime =  ros::Time::now().sec; 
						nh->setParam("lastZoomedScanTime",lastZoomedScanTime);
						zoomOut();
					}else{
						lastFullScanTime =  ros::Time::now().sec; 
						nh->setParam("lastFullScanTime",lastFullScanTime);
					}
					//zoomedScan = false;
					queenSearchCountdown--;
					if (inspectedCellEnable && lateInspect) state = GOTO_CELL;	
				}
				ROS_INFO("Scan pos %i %i %.3f %.3f ",zoomedScan,scanPositionIndex,scanPositionArray.pos[scanPositionIndex].x,scanPositionArray.pos[scanPositionIndex].y);
				if (zoomedScan) scanPositionIndexZoom = scanPositionIndex; else scanPositionIndexUnzoom = scanPositionIndex;
			}
		}
		gotoRelative(0,0,0);
	}
	if (state == SCAN_CELL){
		ros::Duration d =  ros::Time::now() - cellInspectionStart;
		if (cellScanFrameCount++%30 == 0) cellPub.publish(msg);
		if (d.sec > inspectedCellDuration){
			state = SCAN_INIT;
			if (inspectedCellEnable && lateInspect == false){
				zoomOut();
				state = SCAN_MOVE;
				scanFirstMove = true;
			}
			cellScanFrameCount = 0;
		}
	}
	if (state == TRACK_QUEEN){
		queenPub.publish(msg);
		publishWorkersStart = ros::Time::now(); 
	}
	if (forceZoom){
		zoomedPub.publish(msg);
	}

	/*managing randomised publishing*/
	ros::Duration publishWorkersTime = ros::Time::now() - publishWorkersStart;
	if (state != TRACK_QUEEN && publishWorkersTime.sec < workersPublishInterval && publishWorkersTime.sec > 0) workersPub.publish(msg);
	if (publishWorkersTime.sec > workersPublishInterval){
		ros::Duration nextPublishWait((int)((float)rand()/RAND_MAX*workersWaitInterval));
		publishWorkersStart = ros::Time::now() + nextPublishWait; 
		ROS_INFO("Next time of random publishing is %i seconds from now",nextPublishWait.sec); 
	}
	if (reportImageCountdown < 0){
		reportPub.publish(msg);
		reportImageCountdown = 10;
	}
}

void lensCallback(const rr_msgs::LensStatus::ConstPtr &msg)
{
	currentFocus = msg->focus;
	currentZoom = msg->zoom;
       	
	if (msg->zoom == desiredZoom && msg->focus == desiredFocus) cameraFocused = true; else cameraFocused = false;
    if (fabs(msg->zoom-zoomInVal) < fabs(msg->zoom-zoomOutVal))
    {
        minQueenSize = 1000;
        maxQueenSize = 6000;
        cameraZoomed = true;
    }else{  
        minQueenSize = 400;
        maxQueenSize = 1000;
        cameraZoomed = false;
    }
	//ROS_INFO("Lens status %i %i %i",msg->zoom,msg->focus,cameraZoomed);
}

void cameraInfoCallback(const sensor_msgs::CameraInfo::ConstPtr &msg)
{
    if(msg->K[0] == 0)
    {
        ROS_ERROR_ONCE("ERROR: Camera is not calibrated!");
        return;
    }
    for (int i = 0;i<9;i++) intrinsic.at<float>(i/3,i%3) = msg->K[i];
    for (int i = 0;i<msg->D.size();i++)distortion.at<float>(i) = msg->D[i];
    distortionKnown = true;
}


void actionServerCB(const rr_xy_operator::xy_operatorGoalConstPtr &goal, Server *serv)
{
	//state = DOCK_XY;
	// result.success = false;
	while(state != IDLE)
	{
		if(server->isPreemptRequested()){
			state = PREEMPTED;
			server->setPreempted(result);
			state = IDLE;
		}
		if (state == FAILED)
		{
			state = IDLE;
		}
		usleep(10000);
	}
	twist.linear.x = twist.linear.y = twist.linear.z = twist.angular.y = twist.angular.x = 0.0;	
	xyCmd.publish(twist);
}

void otherXYQueenCallback(const rr_msgs::BeePositionArrayConstPtr& msg)
{
}

void otherXYStateCallback(const std_msgs::StringConstPtr& msg)
{
    otherStateStr = msg->data; 
    if (calibrated && state != TRACK_QUEEN && forceInspect == false) /*&& zoomedScan == false && state != GOTO_CELL && state != SCAN_CELL && state != SCAN_INIT && state != SCAN_ADJUST*/
    { 
        if (state != MIRROR && msg->data == "Track-queen" && idleTime < idleMirrorTime && otherLastQueenDetection > lastQueenDetection.sec && (ros::Time::now().sec - otherLastQueenDetection) < mirrorWaitTime){
            state = MIRROR;
            zoomOut(); 
            ROS_INFO("Switched to mirroring other table");
	    configureDetection(minQueenMarkerSize,maxQueenMarkerSize,1);
        }
        if (state == MIRROR && msg->data != "Track-queen"){
            initiateScan(false);
            ROS_INFO("End mirroring, initiating new scan");
        }
        if (state == MIRROR && idleTime > idleMirrorTime){
            initiateScan(true);
            ROS_INFO("End mirroring due to queen resting");
        }
    }
}

void otherXYPositionCallback(const geometry_msgs::PoseStampedConstPtr& msg)
{
	float mX = msg->pose.position.x; 
	float mY = msg->pose.position.y;
	if (state == MIRROR){
		float dx =  tagCameraX[1] - mX - cameraPositionX;  
		float dy =  mY - cameraPositionY;  
		gotoRelative(dx,dy,fabs(dx),fabs(dy));
	}

	float dx = mX - anchorX;
	float dy = mY - anchorY;
	if (sqrt(dx*dx + dy*dy) > 0.01){
		anchorX = mX;
		anchorY = mY;
		idleTime = 0;
	}else{
		idleTime += 0.05;
	}
}

void xyPositionCallback(const geometry_msgs::PoseStampedConstPtr& msg)
{
	//	ROS_INFO("Motor pos %f %f", motorX, motorY);
	motorX = msg->pose.position.x; 
	motorY = msg->pose.position.y;
	 

	if (state == CALIB_ADJUST || state == SCAN_ADJUST) adjustStuckCount++;
	if (state == DOCK){
		stuckCount++;
	}else if (positionReached() == false) stuckCount++;
	if (stuckCount > 100) ROS_INFO("Stuck %i to %s",stuckCount,stateToStr(state));
	if (lastMotorX != motorX || lastMotorY != motorY) stuckCount = 0;
	if (stuckCount > 500 || (stuckCount > 200 && state != LEAVE_DOCK && state != DOCK && state != SCAN_ADJUST && state != CALIB_ADJUST) ){
		ROS_WARN("Table is stuck %f %f",requiredPositionY,motorY);
		rr_xy_operator::ResetPosition srv;
		if (requiredPositionX > motorX) srv.request.x = 0.58; else srv.request.x = 0.000;
		if (requiredPositionY > motorY) srv.request.y = 0.61; else srv.request.y = 0.000;
		if (requiredPositionY == motorY) srv.request.y = motorY; 
		if (requiredPositionX == motorX) srv.request.x = motorX; 
		resetPositionClient.call(srv);
		stuckCount = 0;
		state = DOCK;
	}
	if (adjustStuckCount > 200){
		ROS_WARN("No marker visible when adjusting position, resetting");
		rr_xy_operator::ResetPosition srv;
		srv.request.x = 0.58;
		srv.request.y = 0.61;
		resetPositionClient.call(srv);
		adjustStuckCount = 0;
		state = DOCK;
	}
	lastMotorX = motorX;
	lastMotorY = motorY;
	float mX = motorX;
	float mY = motorY;
	applyHomography(homDirect,&mX, &mY);
	cameraPositionX = mX;
	cameraPositionY = mY;
	//ROS_INFO("State %s, camera position %f %f, total travelled %f %f", stateToStr(state),cameraPositionX, cameraPositionY,travelledX,travelledY);

	if (firstPositionRead){
		travelledX += fabs(lastCameraPositionX - cameraPositionX);
		travelledY += fabs(lastCameraPositionY - cameraPositionY);
		totalTravelledX += fabs(lastCameraPositionX - cameraPositionX);
		totalTravelledY += fabs(lastCameraPositionY - cameraPositionY);
	}
	firstPositionRead = true;
	lastCameraPositionX = mX;
	lastCameraPositionY = mY;

	if (state == GOTO_CELL){
		zoomIn();
		gotoAbsolute(inspectedCellX,inspectedCellY);
		if (positionReached()){
			cellInspectionStart = ros::Time::now();
			state = SCAN_CELL;
			cellScanFrameCount = 0;
		}
	}

	/*docks to the bottom left by info from the end switches*/
	if (state == DOCK)
	{
		calibrated = false;
		queenSearchCountdown =  queenSearchCountdownInit;
		zoomOut();
		/*if (stuckCount > 200){
		  rr_xy_operator::ResetPosition srv;
		  srv.request.x = 0.5;
		  srv.request.y = 0.5;
		  resetPositionClient.call(srv);
		  ROS_INFO("Attempting to resolve stuck table");
		  }*/
		if (switchRight && switchBottom && state == DOCK)
		{
			state = LEAVE_DOCK; 
			ROS_INFO("Docking is done.");
		}
		gotoAbsolute(0,0);
	}
	if (state == LEAVE_DOCK){
		calibrated = false;
		gotoAbsolute(firstTagX,firstTagY,0.01);
		if (positionReached()){
			state = CALIB_ADJUST;
			configureDetection(minCalibrationMarkerSize,maxCalibrationMarkerSize,1);
			tagNumber = 0;
			ROS_INFO("Undocking is done %f %f.",motorX,motorY);
		}
	}

	/*moves between the calibration patterns by odometry*/
	if (state == CALIB_MOVE)
	{
		float xx = tagCameraX[tagNumber]+firstTagX;
		float yy = tagCameraY[tagNumber]+firstTagY;
		gotoAbsolute(xx,yy,0.05);
		ROS_DEBUG("Moving to %f %f ",requiredPositionX,requiredPositionY);	
		if (positionReached()){
			state = CALIB_ADJUST;
			configureDetection(minCalibrationMarkerSize,maxCalibrationMarkerSize,1);
		}
	}

	/*moves between the calibration patterns by odometry*/
	if (state == QUEEN_SEARCH)
	{
		gotoAbsolute(sweepPointX[sweepStep],sweepPointY[sweepStep],0.02);
		if (positionReached(0.01))
		{
			if (++sweepStep > 6) state = MIRROR;
		}
	}


	/*moves to the first tag by odometry*/
	if (state == SCAN_INIT)
	{
		zoomOut();
		tagNumber = 4;
		ROS_DEBUG("Scan init %.3f %.3f %.3f %.3f",cameraPositionX,cameraPositionY,tagCameraX[0],tagCameraY[0]);
		//queenSearchCountdown =  queenSearchCountdownInit;
		gotoAbsolute(0.0,0.0,0.05);
		if (positionReached())
		{
			state = SCAN_ADJUST;
			configureDetection(minCalibrationMarkerSize,maxCalibrationMarkerSize,1);
			adjustCount = 0;
			tagMeasurementX = 0;
			tagMeasurementY = 0;
			scanCount = maxScanCount++;
		}
	}

	/*perfoms a scan step by odometry*/
	if (state == SCAN_MOVE)
	{
		ROS_DEBUG("CP %.3f %.3f %.3f %.3f",cameraPositionX,cameraPositionY,scanPositionArray.pos[scanPositionIndex].x,scanPositionArray.pos[scanPositionIndex].y);
		gotoAbsolute(scanPositionArray.pos[scanPositionIndex].x,scanPositionArray.pos[scanPositionIndex].y,scanPositionArray.pos[scanPositionIndex].v);
		if (positionReached()) state = SCAN_WAIT;
	}


	geometry_msgs::PoseStamped cameraPosition = *msg;
	cameraPosition.pose.position.x = cameraPositionX ; 
	cameraPosition.pose.position.y = cameraPositionY;
	cameraPositionPub.publish(cameraPosition);
}

void xyStatusCallback(const rr_msgs::ActuatorStatusConstPtr& msg)
{
	static int switchTopCount =0;
	static int switchRightCount =0;
	static int switchBottomCount =0;
	static int switchLeftCount =0;
	static int threshold = 10;
	//currentX = currentX*0.9+msg->current_x*0.1;
	//currentY = currentY*0.9+msg->current_y*0.1;
	//ROS_DEBUG("CURRENT; %.3f %.3f %.3f %.3f %.3f %.3f",msg->current_x,msg->current_y,msg->velocity_x,msg->velocity_y,currentX,currentY);
	switchLeft = msg->switchLeft;
	switchTop = msg->switchTop;
	switchBottom = msg->switchBottom;
	switchRight = msg->switchRight;
	//printf("%f %f, %f %f, %f %f\n",cameraPositionX,cameraPositionY,requiredPositionX,requiredPositionY,motorX,motorY);
	if (switchTop) switchTopCount++; else switchTopCount = 0;
	if (switchRight) switchRightCount++; else switchRightCount = 0;
	if (switchBottom) switchBottomCount++; else switchBottomCount = 0;
	if (switchLeft) switchLeftCount++; else switchLeftCount = 0;
	if (switchLeftCount > threshold || switchRightCount > threshold || switchTopCount >  threshold || switchBottomCount > threshold){ 
		if (state != DOCK){
			ROS_WARN("End switch triggered at %f %f! Left: %i Right %i Top %i Bottom %i. Initiate docking procedure",motorX,motorY,switchLeft,switchRight,switchTop,switchBottom);
			if (state != LEAVE_DOCK){
				for (int i = 0;i<9;i++){
					homDirect[i] = homInverse[i] = homEye[i]; 
				}
				/*this mitigates the situation when the table drifted too much and is stuck*/
				if (state == CALIB_ADJUST){
					rr_xy_operator::ResetPosition srv;
					srv.request.x = 0.2; 
					srv.request.y = 0.2;
					if (!resetPositionClient.call(srv)){
						ROS_ERROR("Failed to call position reset service.");
					}else{
						ROS_INFO("Table offset was %.5f %5f.",srv.request.x-srv.response.xMeasure,srv.request.y-srv.response.yMeasure);
					}
				} 
				state = DOCK;
			}
		}
	}
	std_msgs::String stat;
	stat.data = stateStr[state];
	statePub.publish(stat);
}

void beePositionCallback(const rr_msgs::BeePositionArrayConstPtr& msg)
{
	markers = *msg;
	int number = msg->positions.size();

	if (number > 0 &&   msg->positions[0].size < maxQueenSize  &&  msg->positions[0].size > minQueenSize) trackingHits++; else trackingHits=0;
	if (((trackingHits > 0 && state == QUEEN_SEARCH) || trackingHits > 1) && state != DOCK && state != CALIB_MOVE && state != CALIB_ADJUST && state != GOTO_CELL && state != SCAN_CELL){
		if (state != TRACK_QUEEN){
			if (queenSearchCountdown <= 0 || state == QUEEN_SEARCH || state == MIRROR){
				queenTrackingStart = ros::Time::now();
				ros::Duration d = queenTrackingStart - nodeStart;
				ROS_INFO("Queen spotted, starting tracking. Total tracking time %i out of %i. Tracking ratio %02i%%.",trackingDuration,d.sec,100*trackingDuration/d.sec);
				state = TRACK_QUEEN;
				currentTrackLength = -1;
			}else{
				ROS_INFO("Queen spotted when %s, but still scanning - search countdown is %i and focused scan is %i",stateStr[state],queenSearchCountdown,zoomedScan);
			}
		}
	}
	if (state == TRACK_QUEEN )
	{
		if (number > 0 && msg->positions[0].size < maxQueenSize  &&  msg->positions[0].size > minQueenSize) {
			float queenX = -msg->positions[0].info.position.position.y;
			float queenY = +msg->positions[0].info.position.position.z;
			/*if (cameraZoomed){
				queenX = queenX*2.5;	//TODO fix this shit 
				queenY = queenY*2.5; 
			}*/
			ROS_DEBUG("Queen spotted: %0.4f %0.4f Size: %i", queenX, queenY, msg->positions[0].size);
			//ROS_INFO("%.3f %.3f %.3f %.3f",mX,mY,tagX,tagY);
			float cur_bee_x = cameraPositionX + queenX;
			float cur_bee_y = cameraPositionY + queenY;

			if (currentTrackLength == -1){
				lastQueenPositionX = cur_bee_x;
				lastQueenPositionY = cur_bee_y;
				currentTrackLength = 0;
			}

			rr_msgs::BeePosition queenPos;
			queenPos.u = msg->positions[0].u;
			queenPos.v = msg->positions[0].v;
			queenPos.x = cur_bee_x;
			queenPos.y = cur_bee_y;
			markers.positions.push_back(queenPos);

			queenPos.x = cameraPositionX;
			queenPos.y = cameraPositionY;
			markers.positions.push_back(queenPos);
			queenPos.x = queenX;
			queenPos.y = queenY;
			markers.positions.push_back(queenPos);

			queenPositionPub.publish(markers);
			if (fabs(queenX) > 0.001 || fabs(queenY) > 0.001) gotoRelative(queenX,queenY,fabs(queenX),fabs(queenY));
			trackingMisses=0;
			lastQueenDetection = ros::Time::now();

			float dx = (lastQueenPositionX-cur_bee_x);
			float dy = (lastQueenPositionY-cur_bee_y);
			float d = sqrt(dx*dx+dy*dy); 
			if (d > 0.01){
				lastQueenPositionX = cur_bee_x;
				lastQueenPositionY = cur_bee_y;
				currentTrackLength += d;
				totalTrackLength += d;
			} 
		}else{
			if (trackingMisses++ == 0) queenTrackingEnd = ros::Time::now();
		}
		if ((ros::Time::now() -lastQueenDetection).sec > trackingWaitTime){
			ros::Duration dNow = queenTrackingEnd - queenTrackingStart;
			ros::Duration dTotal = queenTrackingEnd - nodeStart;
			trackingDuration += dNow.sec;
			ROS_INFO("Queen not seen for a while, reinitialising scan. Last tracking time %i, total tracking time %i out of %i. Tracking ratio %02i%%.",dNow.sec,trackingDuration,dTotal.sec,100*trackingDuration/dTotal.sec);
			initiateScan(false);
			if (inspectedCellEnable && lateInspect) state = GOTO_CELL;	
		}
	}
}

void publishQueenStats(const ros::TimerEvent&)
{
	rr_msgs::QueenDetectionStatistics stat;
	stat.totalDetectionDuration = trackingDuration;
	if (state == TRACK_QUEEN ) stat.totalDetectionDuration += (ros::Time::now() - queenTrackingStart).sec;
	stat.hive = myHive;
	stat.side = mySide;
	stat.totalRunTime =  (ros::Time::now() - nodeStart).sec;
	stat.detectionRatio = (float)stat.totalDetectionDuration/stat.totalRunTime;
	stat.lastQueenDetection = lastQueenDetection.sec; 
	stat.currentTrackLength = currentTrackLength;
	stat.totalTrackLength = totalTrackLength;
	stat.currentRestingDuration = stat. totalRestingDuration = stat.totalRestingDuration = stat.currentWalkDuration = -1;
	stat.extraArrayInt.push_back((int)forceZoom); 
	/*
	   int32 currentRestingDuration 
	   int32 totalRestingDuration
	   int32 currentWalkDuration
	   int32 totalWalkDuration 
	   */
	queenDetectionStatisticsPub.publish(stat);
	reportImageCountdown--; 
}

/* dynamic reconfigure of crop parameters threshold bee marker size */
void callback(rr_xy_operator::xy_operatorConfig &config, uint32_t level)
{
    ROS_INFO("Reconfigure Request: ");
    queenSearchCountdownInit = config.queenSearchCountdown;	//how many full scans before searching for the queen 
    inspectedCellEnable = config.inspectedCellEnable;	//should we scan the cell? 
    inspectedCellX = config.inspectedCellX;	//coordinates of the inspected cell 
    inspectedCellY = config.inspectedCellY;	//coordinates of the inspected cell
    inspectedCellDuration = config.inspectedCellDuration;	//coordinates of the inspected cell
    trackingWaitTime = config.trackingWaitTime;	//how long to wait when lost track of the queen 
    mirrorWaitTime = config.mirrorWaitTime;	//how long to wait when lost track of the queen 
    idleMirrorTime = config.idleMirrorTime;	//how long to wait mirroring a sleeping queen before doing a cell inspection 
    if (config.forceInspect == true) state = GOTO_CELL;
    if (config.forceInspect == false && (state == SCAN_CELL || state == GOTO_CELL)) state = SCAN_INIT;
    forceInspect = config.forceInspect; //TODO force inspect should be disabled after scan
    if (config.zoomWait && forceZoom == false)
    {
        zoomIn();
        zoomWait = config.zoomInWaitDuration*30;
        forceZoom = true; 
    }
    if (config.zoomWait == false && forceZoom){
        zoomOut();
        zoomWait = 0;
        forceZoom = false;
    }
    ROS_INFO("X: %.3f Y: %.3f\n",inspectedCellX,inspectedCellY);
}

void initiateQueenSearch()
{
    ROS_INFO("Switching to queen search");
    state = QUEEN_SEARCH;
    float minDist = 100;
    sweepStep = 0;
    int index0,index1;
    int offset = 0;
    if (motorY > 0.33) offset = 4;
    ROS_INFO("SEARCH Y %.3f %.3f %i",motorY,(tagCameraY[0] + tagCameraY[2])/2+firstTagY,offset);
    for (int i = 0;i<4;i++){
        float dx = outlineX[i+offset] - motorX; 	
        float dy = outlineY[i+offset] - motorY;	
        if (fmin(dx*dx,dy*dy) < minDist){
            minDist = fmin(dx*dx,dy*dy);
            sweepPointX[0] = searchPointX[i+offset];
            sweepPointY[0] = searchPointY[i+offset];
            index0 = i+offset;
        }
    }
    minDist = 100;
    for (int i = 0;i<4;i++){
        if (i != index0-offset){
            float dx = outlineX[i+offset] - motorX; 	
            float dy = outlineY[i+offset] - motorY;	
            if (fmin(dx*dx,dy*dy) < minDist){
                minDist = fmin(dx*dx,dy*dy);
                sweepPointX[1] = searchPointX[i+offset];
                sweepPointY[1] = searchPointY[i+offset];
                index1 = i+offset;
            }
        }
    }


    for (int i = 0;i<4;i++){
        sweepPointX[i+2] = searchPointX[i+offset];
        sweepPointY[i+2] = searchPointY[i+offset];
    }
    sweepPointX[6] = searchPointX[offset];
    sweepPointY[6] = searchPointY[offset];
    for (int i = 0;i<7;i++) ROS_INFO("Search X: %.2f Y %.2f ",sweepPointX[i],sweepPointY[i]);

}

void otherXYStatsCallback(const rr_msgs::QueenDetectionStatisticsConstPtr& msg)
{
	otherLastQueenDetection = msg->lastQueenDetection;
	if (state == TRACK_QUEEN && otherStateStr == "Track-queen" && lastQueenDetection.sec < msg->lastQueenDetection){
		state = MIRROR;
		trackingDuration += (queenTrackingEnd - queenTrackingStart).sec;
		ROS_INFO("Other table sees the queen better, mirroring it");
		configureDetection(minQueenMarkerSize,maxQueenMarkerSize,1);
	}

	if (state == MIRROR && otherStateStr == "Track-queen" && (ros::Time::now().sec - msg->lastQueenDetection) > mirrorWaitTime)
	{
		initiateQueenSearch();
	}
}

int main(int argc, char** argv)
{
	ros::init(argc, argv, "xy_operator");
	nh = new ros::NodeHandle();
	it = new image_transport::ImageTransport(*nh);
	srand(ros::Time::now().nsec);

	/*read the true tag positions*/
	nh->param("operator/pt0_x", tagCameraX[0], 0.0f);
	nh->param("operator/pt0_y", tagCameraY[0], 0.0f);
	nh->param("operator/pt1_x", tagCameraX[1], 0.494f);
	nh->param("operator/pt1_y", tagCameraY[1], 0.0f);
	nh->param("operator/pt2_x", tagCameraX[2], 0.495f);
	nh->param("operator/pt2_y", tagCameraY[2], 0.393f);
	nh->param("operator/pt3_x", tagCameraX[3], 0.0f);
	nh->param("operator/pt3_y", tagCameraY[3], 0.395f);
	nh->param("operator/zoomIn", zoomInVal,50);
	nh->param("operator/focusIn", focusInVal,50);
	nh->param("operator/irisIn", irisInVal,13);
	nh->param("operator/zoomOut", zoomOutVal,50);
	nh->param("operator/focusOut", focusOutVal,50);
	nh->param("operator/irisOut", irisOutVal,13);
	nh->param("lastFullScanTime", lastFullScanTime,0);
	nh->param("lastZoomedScanTime", lastZoomedScanTime,0);
	ROS_INFO("Scan times %i %i",lastFullScanTime,lastZoomedScanTime);
	generateScanPositions(0.065,true,0.005);
	desiredZoom = zoomOutVal; 
	desiredFocus = focusOutVal; 
	//ROS_INFO("SSS: %f %f %i %i",tagCameraX[2],tagCameraY[2],zoomInVal,zoomOutVal);

	/*establish which side I am on*/
	if (std::string::npos == ros::this_node::getNamespace().find("xy_0")) mySide = 1; 
	if (std::string::npos == ros::this_node::getNamespace().find("xy_1")) mySide = 0;
	for (int i = 0;i<10;i++){
		char hiveName[50];
		sprintf(hiveName,"hive_%i",i);
		if (std::string::npos != ros::this_node::getNamespace().find(hiveName)) myHive = i; 
	}
	char nameSpace[50];
	strcpy(nameSpace,ros::this_node::getNamespace().c_str());
	nameSpace[strlen(nameSpace)-1] = 0;
	sprintf(otherXYPositionName,"%s%i/camera_position",nameSpace,1-mySide);
	sprintf(otherXYQueenName,"%s%i/queen_position",nameSpace,1-mySide);
	sprintf(otherXYStateName,"%s%i/state",nameSpace,1-mySide);
	sprintf(otherXYStatsName,"%s%i/queen_detection_statistics",nameSpace,1-mySide);

	tagCameraX[4] = tagCameraX[0];
	tagCameraY[4] = tagCameraY[0];
	for (int i = 0;i<5;i++){
		tagMotorX[i] = tagCameraX[i];
		tagMotorY[i] = tagCameraY[i];
	}
	nodeStart = ros::Time::now();

	/* Initiate action server */
	server = new Server (*nh, "xy_operator", boost::bind(&actionServerCB, _1, server), false);
	server->start();

	//ROS_INFO("Movement limits: %.3f %.3f %.3f %.3f",borderMinX,borderMinY,borderMaxX,borderMaxY);

	/* Initiate dynamic reconfiguration */
	dynamic_reconfigure::Server<rr_xy_operator::xy_operatorConfig> server;
	dynamic_reconfigure::Server<rr_xy_operator::xy_operatorConfig>::CallbackType f = boost::bind(&callback, _1, _2);
	server.setCallback(f);

	lensZoomPub = nh->advertise<rr_msgs::LensZoom>("lens_controller/set_zoom",1);
	lensFocusPub = nh->advertise<rr_msgs::LensFocus>("lens_controller/set_focus",1);
	cameraPositionPub = nh->advertise<geometry_msgs::PoseStamped>("camera_position",1);
	snapshotPub = it->advertise("comb_snapshot", 1);
	zoomedSnapshotPub = it->advertise("comb_snapshot_zoomed", 1);
	cellPub = it->advertise("cell_snapshot", 1);
	queenPub = it->advertise("queen_image", 1);
	workersPub = it->advertise("workers_image", 1);
	zoomedPub = it->advertise("zoomed_image", 1);
	reportPub = it->advertise("report_image", 1);
	camInfoPub = nh->advertise<sensor_msgs::CameraInfo>("zoom_camera/camera_info", 1);
	xyCmd = nh->advertise<sensor_msgs::JointState>("xy_cmd", 1);
	queenPositionPub = nh->advertise<rr_msgs::BeePositionArray>("queen_position", 1);
	statePub = nh->advertise<std_msgs::String>("state", 1);
	queenDetectionStatisticsPub = nh->advertise<rr_msgs::QueenDetectionStatistics>("queen_detection_statistics", 1);


	resetPositionClient = nh->serviceClient<rr_xy_operator::ResetPosition>("rr_xy_reset");
	focusHomeClient = nh->serviceClient<std_srvs::Empty>("lens_controller/focus_homing");
	zoomHomeClient  = nh->serviceClient<std_srvs::Empty>("lens_controller/zoom_homing");

	beePositionSub = nh->subscribe("whycon/bee_position", 1, beePositionCallback);
	markerSub = nh->subscribe("whycon/markers", 1, tagPositionCallback);
	xyPositionSub = nh->subscribe("xy_position", 1, xyPositionCallback);
	xyStatusSub = nh->subscribe("xy_status", 1, xyStatusCallback);
	imageSub = it->subscribe( "camera/image_mono", 1,imageCallback);
	camInfoSub = nh->subscribe("camera/camera_info", 1, cameraInfoCallback);
	lensStatusSub = nh->subscribe("lens_controller/status", 1, lensCallback);
	ROS_INFO("A, %s %s %s %s",otherXYPositionName,otherXYQueenName,otherXYStateName,otherXYStatsName);
	otherXYPositionSub = nh->subscribe(otherXYPositionName,1,otherXYPositionCallback);
	otherXYQueenSub = nh->subscribe(otherXYQueenName,1,otherXYQueenCallback);
	otherXYStateSub = nh->subscribe(otherXYStateName,1,otherXYStateCallback);
	otherXYStatsSub = nh->subscribe(otherXYStatsName,1,otherXYStatsCallback);

	calculateHomography(tagMotorX, tagMotorY, tagCameraX, tagCameraY);
	queenPublishTimer = nh->createTimer(ros::Duration(1.0), publishQueenStats);
	state = DOCK;
	ros::spin();
	return 0;
}

