/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

WhyCode marker detector

Copyright 2025 Jiri Ulrich, Tomas Krajnik 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




#ifndef WHYCON_ROS_CWHYCONROSNODE_H
#define WHYCON_ROS_CWHYCONROSNODE_H

#include <vector>

#include <ros/ros.h>
#include <image_transport/image_transport.h>
#include <dynamic_reconfigure/server.h>
#include <tf2_ros/transform_broadcaster.h>

#include <std_msgs/Float32MultiArray.h>

#include "whycon/whyconConfig.h"
#include "whycon/SelectMarker.h"
#include "whycon/SetCalibMethod.h"
#include "whycon/SetCalibPath.h"
#include "whycon/SetCoords.h"
#include "whycon/SetDrawing.h"
#include "whycon/GetGuiSettings.h"

#include "whycon/whycon.h"

#include "rr_msgs/BeePosition.h"
#include "rr_msgs/BeePositionArray.h"


namespace whycon_ros
{

class CWhyconROSNode
{

    public:
        bool getGuiSettingsCallback(whycon::GetGuiSettings::Request &req, whycon::GetGuiSettings::Response &res);

        bool setDrawingCallback(whycon::SetDrawing::Request& req, whycon::SetDrawing::Response& res);

        bool setCoordsCallback(whycon::SetCoords::Request& req, whycon::SetCoords::Response& res);

        bool setCalibMethodCallback(whycon::SetCalibMethod::Request& req, whycon::SetCalibMethod::Response& res);

        bool setCalibPathCallback(whycon::SetCalibPath::Request& req, whycon::SetCalibPath::Response& res);

        bool selectMarkerCallback(whycon::SelectMarker::Request& req, whycon::SelectMarker::Response& res);

        void reconfigureCallback(whycon::whyconConfig& config, uint32_t level);

        void cameraInfoCallback(const sensor_msgs::CameraInfoConstPtr& msg);

        void imageCallback(const sensor_msgs::ImageConstPtr& msg);

        CWhyconROSNode();

        ~CWhyconROSNode();

    private:
        
        ros::Subscriber cam_info_sub_;          // camera info subscriber
        image_transport::Subscriber img_sub_;   // camera image raw subscriber

        image_transport::Publisher img_pub_;    // image publisher for GUI
        image_transport::Publisher sus_pub_;    // image publisher to indicate potential false positives
        ros::Publisher markers_pub_;            // publisher of MarkerArray
        ros::Publisher visual_pub_;             // publisher of MarkerArray for RVIZ
        ros::Publisher bee_position_pub_;
        ros::Publisher homography_pub_;
        ros::Publisher debug_pub_;

        bool draw_coords_;
        ros::ServiceServer drawing_srv_;
        ros::ServiceServer calib_path_srv_;
        ros::ServiceServer coord_system_srv_;
        ros::ServiceServer calib_method_srv_;
        ros::ServiceServer select_marker_srv_;
        ros::ServiceServer gui_settings_srv_;
        
        bool publish_visual_;   // whether to publish visualization msgs
        bool use_gui_;          // generate images for graphic interface?
        whycon::CWhycon whycon_;        // WhyCon instance
        whycon::CRawImage *image_;      // image wrapper for WhyCon
        double circle_diameter_;  // marker diameter [m]
        double min_corner_size_;

        std::vector<whycon::SMarker> whycon_detections_;    // array of detected markers
        
        std::vector<float> intrinsic_mat_;        // intrinsic matrix from camera_info topic
        std::vector<float> distortion_coeffs_;    // distortion parameters from camera_info topic

        dynamic_reconfigure::Server<whycon::whyconConfig> dyn_srv_;
        dynamic_reconfigure::Server<whycon::whyconConfig>::CallbackType dyn_srv_cb_;

        bool identify_;
        bool publish_tf_;
        tf2_ros::TransformBroadcaster tf_broad_;

        ros::Time lastQueenDetection;
        int suspicious_detection_timeout;

};

}


#endif
