/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

WhyCode marker detector

Copyright 2025 Jiri Ulrich, Tomas Krajnik 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




#ifndef WHYCON__SSTRUCTDEFS_H
#define WHYCON__SSTRUCTDEFS_H

namespace whycon
{

typedef struct
{
    float angle;    // axis rotation angle
    int id;         // marker decoded ID
    int edgeIndex;  // idx of starting edge
} SDecoded;


// this structure contains information related to image coordinates and dimensions of the detected pattern
typedef struct
{
    float x;                    // center in image coordinates
    float y;                    // center in image coordinates
    float angle, horizontal;    // orientation (not really used in this case, see the SwarmCon version of this software)
    int size;                   // number of pixels
    int maxy, maxx, miny, minx; // bounding box dimensions
    int mean;                   // mean brightness
    int type;                   // black or white ?
    float roundness;            // result of the first roundness test, see Eq. 2 of paper [1]
    float bwRatio;              // ratio of white to black pixels, see Algorithm 2 of paper [1]
    bool round;                 // segment passed the initial roundness test
    bool valid;                 // marker passed all tests and will be passed to the transformation phase
    float m0, m1;               // eigenvalues of the pattern's covariance matrix, see Section 3.3 of [1]
    float v0, v1;               // eigenvectors of the pattern's covariance matrix, see Section 3.3 of [1]
    float r0, r1;               // ratio of inner vs outer ellipse dimensions (used to establish ID, see the SwarmCon version of this class)
    int ID;                     // pattern ID

    int threshold;
    int size_inner;
    int size_outer;
} SSegment;

// which transform to use
typedef enum
{
    TRANSFORM_NONE,     //camera-centric
    TRANSFORM_2D,       //3D->2D homography
    TRANSFORM_3D,       //3D user-defined - linear combination of four translation/rotation transforms
    TRANSFORM_4D,       //3D user-defined - full 4x3 matrix
    TRANSFORM_INV,      //for testing purposes
    TRANSFORM_NUMBER
} ETransformType;

typedef struct
{
    float u, v;                 // real center in the image coords
    float x, y, z, d;           // position and distance in the camera coords
    float roll, pitch, yaw;     // fixed axis angles
    float angle;                // axis angle around marker's surface normal
    float n0, n1, n2;           // marker surface normal pointing from the camera
    float qx, qy, qz, qw;       // quaternion
    // ??? not used float roundness;            // segment roundness as calculated by 5 of [1]
    // ??? not used float bwratio;              // black/white area ratio
    // ??? not used int ID;                     // ID of marker
} STrackedObject;

typedef struct
{
    float u[2];     // retransformed x coords
    float v[2];     // retransformed y coords
    float n[2][3];  // both solutions of marker's surface normal
    float t[2][3];  // both solutions of position vector
} SEllipseCenters;

// rotation/translation model of the 3D transformation                                                                                                                  
typedef struct
{
    float orig[3];      // translation {x, y, z}
    float simlar[9];    // rotation description, similarity transformation matrix
} S3DTransform;

typedef struct
{
    bool valid;
    SSegment seg;
    STrackedObject obj;
} SMarker;

}

#endif
