/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

WhyCode marker detector

Copyright 2025 Jiri Ulrich, Tomas Krajnik 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




#ifndef WHYCON__CWHYCON_H
#define WHYCON__CWHYCON_H

#include <vector>
#include <deque>

#include <stdlib.h>
#include <string>
#include <cmath>

// WhyCon/WhyCode libs
#include "whycon/CTimer.h"
#include "whycon/CCircleDetect.h"
#include "whycon/CTransformation.h"
#include "whycon/CNecklace.h"
#include "whycon/CRawImage.h"

namespace whycon
{

class CWhycon {

    public:

        int image_width_ = 640;         // default camera resolution
        int image_height_ = 480;        // default camera resolution
        float circle_diameter_;         // default black circle diameter [m];
        float field_length_ = 1.0;      // X dimension of the coordinate system
        float field_width_ = 1.0;       // Y dimension of the coordinate system

        // marker detection variables
        bool identify_ = false;     // whether to identify ID
        int num_markers_;           // num of markers to track
        int num_found_ = 0;         // num of markers detected in the last step
        int num_static_ = 0;        // num of non-moving markers

        // marker identification
        int id_bits_;       // num of ID bits
        int id_samples_;    // num of samples to identify ID
        int hamming_dist_;  // hamming distance of ID code

        CWhycon();
        ~CWhycon();
        
        void init(float circle_diam, bool use_gui, int id_b, int id_s, int ham_dist, int markers, int img_w, int img_h);
        void setDrawing(bool draw_coords, bool draw_segments);
        void setCoordinates(ETransformType type);
        void autocalibration();
        void manualcalibration();
        void loadCalibration(std::string& path);
        void saveCalibration(std::string& path);
        void selectMarker(float x, float y);
        void updateConfiguration(bool id, int markers, float diam, float bee_diam, int min_size, int max_size, int corner_size, double fl, double fw, double ict, double fct, double art, double cdtr, double cdta);
        void updateCameraInfo(std::vector<float> &intrinsic_mat, std::vector<float> &distortion_coeffs);
        void processImage(CRawImage *image, std::vector<SMarker> &whycon_detections);
        bool getDrawCoords();
        bool getDrawSegments();
        int getCoordinates();
        bool getCalibrationStatus();
        std::vector<float> getHomographyMatrix();
        void setHomoSquarePoints(const std::vector<float>& pts);

    private:

        // GUI-related stuff
        bool use_gui_;          // whether use graphic interface
        bool draw_coords_;      // draw coordinates at the marker's positions
        bool draw_segments_;    // draw segmentation outcome
        int eval_time_;         // time required to detect the patterns

        CTransformation *trans_;    // transformation instance
        CNecklace *decoder_;        // instance to decode marker's ID

        std::deque<SMarker> current_marker_array_;      // array of currently detected markers
        std::deque<SMarker> last_marker_array_;         // array of previously detected markers
        std::deque<CCircleDetect*> detector_array_;     // array of detector instances for each marker

        bool calibrated_coords_;

        std::array<std::map<int, int>, 4> indices_autocalib;
        int index_autocalib[4];

        // variables related to (auto) calibration
        const int calibration_steps_ = 20;              // how many measurements to average to estimate calibration pattern position (manual calib)
        const int auto_calibration_steps_ = 30;         // how many measurements to average to estimate calibration pattern position (automatic calib)  
        const int auto_calibration_pre_steps_ = 10;     // how many measurements to discard before starting to actually auto-calibrating (automatic calib)  
        int calib_num_ = 5;                             // number of objects acquired for calibration (5 means calibration winished inactive)
        STrackedObject calib_[4];                       // array to store calibration patterns positions
        STrackedObject homo_square_pts_[4];
        std::vector<STrackedObject> calib_tmp_;         // array to store several measurements of a given calibration pattern
        int calib_step_ = calibration_steps_ + 2;       // actual calibration step (num of measurements of the actual pattern)
        
        bool autocalibrate_ = false;                    // is the autocalibration in progress ?
        bool mancalibrate_ = false;

        ETransformType last_transform_type_ = TRANSFORM_2D;     // pre-calibration transform (used to preserve pre-calibation transform type)
        int was_markers_ = 1;                                   // pre-calibration number of makrers to track (used to preserve pre-calibation number of markers to track)

        
        

        /*manual calibration can be initiated by pressing 'r' and then clicking circles at four positions (0,0)(fieldLength,0)...*/
        void manualCalib();

        /*finds four outermost circles and uses them to set-up the coordinate system - [0,0] is left-top, [0,fieldLength] next in clockwise direction*/
        void autoCalib();
};

}

#endif
