/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

Camera driver

Copyright 2025 Jiri Ulrich 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




// Copyright [2015] Takashi Ogura<t.ogura@gmail.com>

#include "cv_camera/capture.h"
#include <sstream>
#include <string>

namespace cv_camera
{

namespace enc = sensor_msgs::image_encodings;

Capture::Capture(ros::NodeHandle &node, const std::string &topic_name,
                 int32_t buffer_size, const std::string &frame_id,
                 const std::string& camera_name)
    : node_(node),
      it_(node_),
      topic_name_(topic_name),
      buffer_size_(buffer_size),
      frame_id_(frame_id),
      info_manager_(node_, camera_name),
      capture_delay_(ros::Duration(node_.param("capture_delay", 0.0))),
      fail_reads_timeout_(11.0)  // 11.0 seconds
{
}

void Capture::loadCameraInfo()
{
  std::string url;
  if (node_.getParam("camera_info_url", url))
  {
    if (info_manager_.validateURL(url))
    {
      info_manager_.loadCameraInfo(url);
    }
  }

  rescale_camera_info_ = node_.param<bool>("rescale_camera_info", false);

  for (int i = 0;; ++i)
  {
    int code = 0;
    double value = 0.0;
    std::stringstream stream;
    stream << "property_" << i << "_code";
    const std::string param_for_code = stream.str();
    stream.str("");
    stream << "property_" << i << "_value";
    const std::string param_for_value = stream.str();
    if (!node_.getParam(param_for_code, code) || !node_.getParam(param_for_value, value))
    {
      break;
    }
    if (!cap_.set(code, value))
    {
      ROS_ERROR_STREAM("Setting with code " << code << " and value " << value << " failed"
                                            << std::endl);
    }
  }
}

void Capture::rescaleCameraInfo(int width, int height)
{
  double width_coeff = static_cast<double>(width) / info_.width;
  double height_coeff = static_cast<double>(height) / info_.height;
  info_.width = width;
  info_.height = height;

  // See http://docs.ros.org/api/sensor_msgs/html/msg/CameraInfo.html for clarification
  info_.K[0] *= width_coeff;
  info_.K[2] *= width_coeff;
  info_.K[4] *= height_coeff;
  info_.K[5] *= height_coeff;

  info_.P[0] *= width_coeff;
  info_.P[2] *= width_coeff;
  info_.P[5] *= height_coeff;
  info_.P[6] *= height_coeff;
}

void Capture::open(int32_t device_id)
{
  cap_.open(device_id);
  if (!cap_.isOpened())
  {
    std::stringstream stream;
    stream << "device_id" << device_id << " cannot be opened";
    throw DeviceError(stream.str());
  }
  pub_ = it_.advertiseCamera(topic_name_, buffer_size_);
  
  create_aux_connections();
  loadCameraInfo();
}

void Capture::open(const std::string &device_path)
{
  cap_.open(device_path, cv::CAP_V4L);
  if (!cap_.isOpened())
  {
    throw DeviceError("device_path " + device_path + " cannot be opened");
  }
  pub_ = it_.advertiseCamera(topic_name_, buffer_size_);

  create_aux_connections();
  loadCameraInfo();
}

void Capture::create_aux_connections()
{

  status_pub_ = node_.advertise<sensor_msgs::Image>(topic_name_ + "_status", 1);
  node_.getParam("lens_srv", lens_srv_name_);
  shutdown_lens_client_ = node_.serviceClient<std_srvs::Empty>(lens_srv_name_);
  shutdown_srv_ = node_.advertiseService("shutdown", &Capture::shutdown_cb, this);
}

bool Capture::shutdown_cb(std_srvs::Empty::Request& request, std_srvs::Empty::Response& response)
{
  ROS_WARN("Shutdown request from lens");
  ros::shutdown();
  return true;
}

void Capture::shutdown_and_notify()
{
  std_srvs::Empty empty_req;
  shutdown_lens_client_.call(empty_req);
  ros::shutdown();
}

void Capture::open()
{
  open(0);
}

void Capture::openFile(const std::string &file_path)
{
  cap_.open(file_path);
  if (!cap_.isOpened())
  {
    std::stringstream stream;
    stream << "file " << file_path << " cannot be opened";
    throw DeviceError(stream.str());
  }
  pub_ = it_.advertiseCamera(topic_name_, buffer_size_);

  std::string url;
  if (node_.getParam("camera_info_url", url))
  {
    if (info_manager_.validateURL(url))
    {
      info_manager_.loadCameraInfo(url);
    }
  }
}

bool Capture::capture()
{
  if (cap_.read(image_))
  {
    switch(rotation_)
    {
      case 90:
        cv::rotate(image_, bridge_.image, cv::ROTATE_90_CLOCKWISE);
        break;
      case 180:
        cv::rotate(image_, bridge_.image, cv::ROTATE_180);
        break;
      case 270:
        cv::rotate(image_, bridge_.image, cv::ROTATE_90_COUNTERCLOCKWISE);
        break;
      default:
        bridge_.image = image_;
        break;
    }

    ros::Time stamp = ros::Time::now() - capture_delay_;
    bridge_.encoding = bridge_.image.channels() == 3 ? enc::BGR8 : enc::MONO8;
    bridge_.header.stamp = stamp;
    bridge_.header.frame_id = frame_id_;

    info_ = info_manager_.getCameraInfo();
    if (info_.height == 0 && info_.width == 0)
    {
      info_.height = bridge_.image.rows;
      info_.width = bridge_.image.cols;
    }
    else if (info_.height != bridge_.image.rows || info_.width != bridge_.image.cols)
    {
      if (rescale_camera_info_)
      {
        int old_width = info_.width;
        int old_height = info_.height;
        rescaleCameraInfo(bridge_.image.cols, bridge_.image.rows);
        ROS_INFO_ONCE("Camera calibration automatically rescaled from %dx%d to %dx%d",
                      old_width, old_height, bridge_.image.cols, bridge_.image.rows);
      }
      else
      {
        ROS_WARN_ONCE("Calibration resolution %dx%d does not match camera resolution %dx%d. "
                      "Use rescale_camera_info param for rescaling",
                      info_.width, info_.height, bridge_.image.cols, bridge_.image.rows);
      }
    }
    info_.header.stamp = stamp;
    info_.header.frame_id = frame_id_;

    reads_time_begin_ = ros::Time::now();

    return true;
  }

  if ((ros::Time::now() - reads_time_begin_) > fail_reads_timeout_) {
    ROS_WARN("Frames not read for more than %f seconds.", fail_reads_timeout_.toSec());
    shutdown_and_notify();
  }

  return false;
}

void Capture::publish()
{
  pub_.publish(*getImageMsgPtr(), info_);

  sensor_msgs::Image msg;
  msg.header = getImageMsgPtr()->header;
  status_pub_.publish(msg);
}

bool Capture::setPropertyFromParam(int property_id, const std::string &param_name)
{
  if (cap_.isOpened())
  {
    double value = 0.0;
    if (node_.getParam(param_name, value))
    {
      ROS_INFO("setting property %s = %lf", param_name.c_str(), value);
      return cap_.set(property_id, value);
    }
  }
  return true;
}

// clockwise rotation
bool Capture::setRotation(int32_t rotation)
{
  switch(rotation)
  {
    case 0:
      rotation_ = 0;
      break;
    case 90:
      rotation_ = 90;
      break;
    case 180:
      rotation_ = 180;
      break;
    case 270:
      rotation_ = 270;
      break;
    default:
      rotation_ = 0;
      return false;
  }

  return true;
}

} // namespace cv_camera
