# Software for the Autonomous Robotic Observation and Behavioral Analysis system
#
# Agent ROS driver example
#
# Copyright 2025 Agent Tomas Roucek 
#
# Commercial use of the software requires written consent of the copyright holders. 
#
# For Open Research and Educational use, the following applies:
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at

#   http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
import rospy
from sensor_msgs.msg import Joy, JointState
from std_msgs.msg import Header

class JoyToJointState:
    def __init__(self):
        # Initialize the ROS node
        rospy.init_node('joy_to_joint_state', anonymous=True)
        self.init = True
        # Initialize joint publisher
        self.joint_pub = rospy.Publisher('/desired_joint_states', JointState, queue_size=10)
        rospy.Subscriber('/hive_0/xy_1/joint_states', JointState, self.pose_callback)
        # Subscribe to the joy topic
        rospy.Subscriber('/joy', Joy, self.joy_callback)
        print("Joy to joint state node started")
        # Initial joint state
        self.joint_state = JointState()
        self.joint_state.header = Header()
        self.joint_state.name = ["joint_2", "joint_1"]
        self.joint_state.position = [0.0, 0.0]
        self.joint_state.velocity = [0.0, 0.0]
        self.j1_min = 0.65
        self.j1_max = 4.2
        self.j2_min = 3.3
        self.j2_max = 1.57*3


    def pose_callback(self, joint_msg):
        if self.init:
            print("init")
            self.joint_state.position[0] = joint_msg.position[0]
            self.joint_state.position[1] = joint_msg.position[1]
            self.joint_state.velocity[0] = joint_msg.velocity[0]
            self.joint_state.velocity[1] = joint_msg.velocity[1]
            self.init = False


    def joy_callback(self, joy_msg):
        if self.init:
            return
        # Assume we are using axis 0 on the joystick to control the joint
        joint_control_axis = joy_msg.axes[0]
        rospy.loginfo(f"Received joystick input: {joint_control_axis}")
        joint_control_axis2 = joy_msg.axes[1]
        rospy.loginfo(f"Received joystick input: {joint_control_axis2}")

        joint_control_axis3 = joy_msg.axes[2]
        rospy.loginfo(f"Received joystick input: {joint_control_axis3}")
        joint_control_axis4 = joy_msg.axes[3]
        rospy.loginfo(f"Received joystick input: {joint_control_axis4}")
        # Update joint position based on joystick input
        self.joint_state.position[0] += joint_control_axis * 0.05  # Increment position by a small value
        self.joint_state.position[1] += joint_control_axis2 * 0.05  # Increment position by a small value
        if self.joint_state.position[0] > self.j1_max:
            self.joint_state.position[0] = self.j1_max
        if self.joint_state.position[0] < self.j1_min:
            self.joint_state.position[0] = self.j1_min
        if self.joint_state.position[1] > self.j2_max:
            self.joint_state.position[1] = self.j2_max
        if self.joint_state.position[1] < self.j2_min:
            self.joint_state.position[1] = self.j2_min
        self.joint_state.velocity[0] = joint_control_axis3# * 0.01  # Increment position by a small value
        self.joint_state.velocity[1] = joint_control_axis4# * 0.01  # Increment position by a small value
        # Publish the updated joint state
        self.joint_state.header.stamp = rospy.Time.now()
        self.joint_pub.publish(self.joint_state)

    def start(self):
        rospy.spin()

if __name__ == '__main__':
    try:
        joy_to_joint = JoyToJointState()
        joy_to_joint.start()
    except rospy.ROSInterruptException:
        pass
