# Software for the Autonomous Robotic Observation and Behavioral Analysis system
#
# Camera optics driver
#
# Copyright 2025 Jiri Ulrich 
#
# Commercial use of the software requires written consent of the copyright holders. 
#
# For Open Research and Educational use, the following applies:
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at

#   http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#!/usr/bin/env python3

from lens_controller_ASH10 import Harrier10xController

import rospy
from rr_msgs.msg import LensZoom, LensFocus, LensStatus, LensIris
from std_srvs.srv import Empty, EmptyResponse


class LensControllerNode:
    def __init__(self):
        self.shutdown = False

        rospy.init_node('rr_lens_controller')

        self.frame_id = rospy.get_param('~frame_id')
        self.lib_path = rospy.get_param('~lib_path')

        self.serial_number = rospy.get_param('~serial_number').strip()
        self.init_zoom = rospy.get_param('~init_zoom', 0)
        self.init_focus = rospy.get_param('~init_focus', 0)
        self.init_iris = rospy.get_param('~init_iris', 10)

        self.camera_srv_name = rospy.get_param('~camera_srv')
        rospy.loginfo(f'Waiting for the camera service {self.camera_srv_name}')
        rospy.wait_for_service(self.camera_srv_name)

        self.controller = Harrier10xController(self.serial_number, self.lib_path)
        self.controller.start()
        rospy.loginfo('Connected to the lens controller')
        rospy.loginfo('Moving to start position')
        self.controller.set_pos_zoom(self.init_zoom)
        self.controller.set_pos_focus(self.init_focus)
        self.controller.set_pos_iris(self.init_iris)
        rospy.loginfo('Ready')

        self.zoom_sub = rospy.Subscriber('~set_zoom', LensZoom, self.set_camera_zoom)
        self.focus_sub = rospy.Subscriber('~set_focus', LensFocus, self.set_camera_focus)
        self.iris_sub = rospy.Subscriber('~set_iris', LensIris, self.set_camera_iris)

        self.lens_homing_srv = rospy.Service('~lens_homing', Empty, self.lens_homing)
        self.shutdown_srv = rospy.Service('~shutdown', Empty, self.shutdown_from_camera)

        self.status_pub = rospy.Publisher('~status', LensStatus, queue_size=1)

    def shutdown_and_notify(self):
        rospy.logwarn('Shutting down')
        self.shutdown = True

    def shutdown_from_camera(self, req):
        rospy.logwarn('Shutdown request from camera received')
        self.shutdown = True
        return EmptyResponse()

    def publish_status(self):
        zoom, focus, iris = self.controller.get_status_zfi()

        msg = LensStatus()
        msg.header.stamp = rospy.Time.now()
        msg.header.frame_id = self.frame_id
        
        if zoom is not None and focus is not None and iris is not None:
            msg.zoom = zoom
            msg.focus = focus
            msg.iris = iris
        else:
            self.shutdown_and_notify()

        self.status_pub.publish(msg)

    def set_camera_zoom(self, msg):
        if not self.controller.set_pos_zoom(msg.zoom):
            self.shutdown_and_notify()

    def set_camera_focus(self, msg):
        if not self.controller.set_pos_focus(msg.focus):
            self.shutdown_and_notify()

    def set_camera_iris(self, msg):
        if not self.controller.set_pos_iris(msg.iris):
            self.shutdown_and_notify()

    def lens_homing(self, req):
        if not self.controller.lens_homing():
            self.shutdown_and_notify()
        return EmptyResponse()

    def is_shutdown(self):
        return rospy.is_shutdown() or self.shutdown


if __name__ == '__main__':
    node = LensControllerNode()

    r = rospy.Rate(1)
    while not node.is_shutdown():
        node.publish_status()
        r.sleep()
