# Software for the Autonomous Robotic Observation and Behavioral Analysis system
#
# Camera optics driver
#
# Copyright 2025 Jiri Ulrich 
#
# Commercial use of the software requires written consent of the copyright holders. 
#
# For Open Research and Educational use, the following applies:
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at

#   http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
import time
import threading
import ctypes as c

import rospy


class HarrierStatus:
    ok = 0
    access_fail = 1
    rx_out_of_bounds = 2
    rx_buffer_overflow = 3
    rx_empty = 4
    open_fail = 5
    bad_index = 6
    bad_handle = 7
    buffer_too_small = 8
    bad_args = 9


class ViscaStatus:
    # custom error codes
    empty_rx = 0xDD
    unaligned = 0xEE
    unknown = 0xFF

    # predefined Visca protocol error codes
    ok = 0x00
    err_msg_len = 0x01
    err_syntax = 0x02
    err_cmd_buf_full = 0x03
    err_cmd_cancelled = 0x04
    err_no_socket = 0x05
    err_cmd_not_exe = 0x41

    def is_known_err(err):
        return err in [ViscaStatus.err_msg_len,
                       ViscaStatus.err_syntax,
                       ViscaStatus.err_cmd_buf_full,
                       ViscaStatus.err_cmd_cancelled,
                       ViscaStatus.err_no_socket,
                       ViscaStatus.err_cmd_not_exe]


class Visca:
    # command
    cmd_rx_sz = 6
    icr_night = [0x81, 0x01, 0x04, 0x01, 0x02, 0xFF]
    initialize_lens = [0x81, 0x01, 0x04, 0x19, 0x01, 0xFF]
    focus_manual = [0x81, 0x01, 0x04, 0x38, 0x03, 0xFF]
    ae_iris_priority = [0x81, 0x01, 0x04, 0x39, 0x0B, 0xFF]

    # inquiry
    inq_zoom_rx_sz = 7
    zoom_inq = [0x81, 0x09, 0x04, 0x47, 0xFF]

    inq_focus_rx_sz = 7
    focus_inq = [0x81, 0x09, 0x04, 0x48, 0xFF]

    inq_iris_rx_sz = 7
    iris_inq = [0x81, 0x09, 0x04, 0x4B, 0xFF]

    def zoom_direct(val):
        return [0x81, 0x01, 0x04, 0x47, (val >> 12) & 0xF, (val >> 8) & 0xF, (val >> 4) & 0xF, val & 0xF, 0xFF]

    def focus_direct(val):
        return [0x81, 0x01, 0x04, 0x48, (val >> 12) & 0xF, (val >> 8) & 0xF, (val >> 4) & 0xF, val & 0xF, 0xFF]

    def iris_direct(val):
        return [0x81, 0x01, 0x04, 0x4B, 0x00, 0x00, (val >> 4) & 0xF, val & 0xF, 0xFF]

    def decode_zoom_inq(buf):
        return (buf[2] << 12) | (buf[3] << 8) | (buf[4] << 4) | buf[5]

    def decode_focus_inq(buf):
        return (buf[2] << 12) | (buf[3] << 8) | (buf[4] << 4) | buf[5]

    def decode_iris_inq(buf):
        return (buf[4] << 4) | buf[5]

    def verify_rx(buf):
        if len(buf) == 0:
            rospy.logdebug('Harrier Visca empty buffer')
            return ViscaStatus.empty_rx
        if not (len(buf) > 2 and buf[0] in [0x90, 0xA0] and buf[-1] == 0xFF):
            rospy.logdebug(f'Harrier Visca unaligned buffer. RX[len={len(buf)}] = {[hex(b) for b in buf]}')
            return ViscaStatus.unaligned
        if buf[1] not in [0x40, 0x41, 0x42, 0x43, 0x50, 0x51, 0x52, 0x53]:
            if buf[1] in [0x60, 0x61, 0x62, 0x63] and ViscaStatus.is_known_err(buf[2]):
                rospy.logdebug(f'Harrier Visca report error. RX[len={len(buf)}] = {[hex(b) for b in buf]}')
                return buf[2]
            rospy.logerr(f'Harrier Visca report UNKNOWN error. RX[len={len(buf)}] = {[hex(b) for b in buf]}')
            return ViscaStatus.unknown
        return ViscaStatus.ok


class HarrierCom:
    def __init__(self, serial_number, lib_path, retry_th=5):
        self.retry_th = retry_th
        self.lock = threading.Lock()
        self.sn = serial_number
        self.harrier_handle = -1
        self.dev_idx = -1

        self.lib = c.CDLL(lib_path)
        self.FT_LIST_BY_INDEX = 0x40000000
        self.FT_OPEN_BY_SERIAL_NUMBER = 0x00000001
        self.FT_CreateDeviceInfoList = self.lib.FT_CreateDeviceInfoList
        self.FT_CreateDeviceInfoList.restype = c.c_uint
        self.FT_CreateDeviceInfoList.argtypes = [c.POINTER(c.c_uint)] 
        self.FT_ListDevices = self.lib.FT_ListDevices
        self.FT_ListDevices.restype = c.c_uint
        self.FT_ListDevices.argtypes = [c.c_void_p, c.c_void_p, c.c_uint]
        self.HarrierCommsUSBOpenByIndex = self.lib.HarrierCommsUSBOpenByIndex
        self.HarrierCommsUSBOpenByIndex.restype = c.c_uint
        self.HarrierCommsUSBOpenByIndex.argtypes = [c.POINTER(c.c_int), c.c_int]
        self.HarrierCommsUSBTransmit = self.lib.HarrierCommsUSBTransmit
        self.HarrierCommsUSBTransmit.restype = c.c_uint
        self.HarrierCommsUSBTransmit.argtypes = [c.c_int, c.c_void_p, c.c_size_t]
        self.HarrierCommsUSBClose = self.lib.HarrierCommsUSBClose
        self.HarrierCommsUSBClose.restype = c.c_uint
        self.HarrierCommsUSBClose.argtypes = [c.c_int]
        self.HarrierCommsUSBReceive = self.lib.HarrierCommsUSBReceive
        self.HarrierCommsUSBReceive.restype = c.c_uint
        self.HarrierCommsUSBReceive.argtypes = [c.c_int, c.c_void_p, c.c_size_t, c.POINTER(c.c_ubyte), c.c_int]

    def connect(self):
        print('##########')
        print(f'Searching for SN: {self.sn}')
        print('----------')
        
        num_devs = c.c_uint()
        status = self.FT_CreateDeviceInfoList(c.byref(num_devs))
        num_devs = num_devs.value
        print(f'List of {num_devs} connected devices. FT_STATUS: {status}')

        dev_idx = -1
        if status == 0 and num_devs > 0:
            for d_idx in range(num_devs):
                d_sn = c.create_string_buffer(32)
                status = self.FT_ListDevices(d_idx, d_sn, self.FT_LIST_BY_INDEX | self.FT_OPEN_BY_SERIAL_NUMBER)
                d_sn = d_sn.value.decode()
                if status == 0:
                    print(f'Device ID: {d_idx} SN: {d_sn}')
                    if d_sn == self.sn:
                        dev_idx = d_idx
                else:
                    print(f'FAILED to query SN of the connected device ID: {d_idx}. FT_STATUS: {status}')
        else:
            print(f'FAILED to enumerate devices. FT_STATUS: {status}')

        print('----------')
        print(f'Device ID: {dev_idx}')
        print('##########')

        if dev_idx != -1:
            self.dev_idx = dev_idx
            self.harrier_handle = c.c_int()
            status = self.HarrierCommsUSBOpenByIndex(c.byref(self.harrier_handle), self.dev_idx)
            if status != HarrierStatus.ok:
                rospy.logerr(f'FAILED to open device. Harrier status: {status}')

    def __del__(self):
        self.disconnect()

    def disconnect(self):
        if self.harrier_handle != -1:
            status = self.HarrierCommsUSBClose(self.harrier_handle)
            if status != HarrierStatus.ok:
                rospy.logerr(f'FAILD to close device. Harrier status: {status}')
            else:
                self.harrier_handle = -1

    def _send(self, tx):  # helper function. DON'T CALL OUTSIDE SELF.LOCK
        c_arr = (c.c_uint8 * len(tx))(*tx)
        status = self.HarrierCommsUSBTransmit(self.harrier_handle, c_arr, len(c_arr))
        if status != HarrierStatus.ok:
            rospy.logdebug(f'Harrier status {status}. tx[{len(tx)}]= {[hex(r) for r in tx]}')
        return status

    def _read(self, rx_size, timeout):  # helper function. DON'T CALL OUTSIDE SELF.LOCK
        rx = (c.c_uint8 * rx_size)()
        rx_sz = c.c_size_t(rx_size)
        rx_sz_exp = c.c_ubyte(rx_size)
        status = self.HarrierCommsUSBReceive(self.harrier_handle, rx, rx_sz, c.byref(rx_sz_exp), timeout)
        rx_sz_exp = rx_sz_exp.value
        rx = [rx[i] for i in range(rx_sz_exp)]
        if status != HarrierStatus.ok:
            rospy.logdebug(f'Harrier status {status}. rx[{len(rx)}]= {[hex(r) for r in rx]}')
        return status, rx

    def _send_and_read(self, tx, rx_size, timeout, sleep):
        with self.lock:
            status_tx = self._send(tx)
            #print(f'Harrier status {status_tx}. tx[{len(tx)}]= {[hex(r) for r in tx]}')
            time.sleep(sleep)
            status_rx, rx = self._read(rx_size, timeout)
            #print(f'Harrier status {status_rx}. rx[{len(rx)}]= {[hex(r) for r in rx]}')
            return status_tx, status_rx, rx

    def empty_rx_buffer(self):
        retry_counter = 0
        while True:

            if rospy.is_shutdown():  # to prevent ros node hangs on ctrl+c
                return False

            retry_counter += 1
            # vicsa inq on power. used as dummy cmd to init connection.
            # without any cmd, the reply buffer isn't available and can't be cleaned.
            tx = [0x81, 0x09, 0x04, 0x00, 0xFF]
            status_tx, status_rx, rx = self._send_and_read(tx, rx_size=100, timeout=2000, sleep=0.1)

            if 4 <= len(rx) < 100 and status_tx == HarrierStatus.ok and status_rx == HarrierStatus.ok:
                # 4 - correct response to the inq. 100 - full buffer capacity reached
                break

            rospy.logwarn(f'Retry \'empty rx\' attempt [{retry_counter}/{self.retry_th}] len={len(rx)} status: tx={status_tx} rx={status_rx}')
            if retry_counter >= self.retry_th:
                rospy.logdebug(f'FAILED to clean the rx buffer after {retry_counter} attempts')
                return False

        return True

    def send_and_read(self, tx, rx_size, timeout=1000, sleep=0.001):
        retry_counter = 0
        rx = None
        while True:
            if rospy.is_shutdown():  # to prevent ros node hangs on ctrl+c
                return False, []

            retry_counter += 1
            status_tx, status_rx, rx = self._send_and_read(tx, rx_size, timeout, sleep)

            status_buf = Visca.verify_rx(rx)
            
            if status_buf == ViscaStatus.ok and status_tx == HarrierStatus.ok and status_rx == HarrierStatus.ok:
                break

            rospy.logwarn(f'Retry \'send&read\' attempt [{retry_counter}/{self.retry_th}] len={len(rx)} status: tx={status_tx} rx={status_rx} buf={hex(status_buf)}')
            if retry_counter >= self.retry_th:
                rospy.logdebug(f'FAILED to send&read cmds after {retry_counter} attempts')
                return False, []

            if status_buf in [ViscaStatus.unaligned, ViscaStatus.empty_rx, ViscaStatus.unknown]:
                status_clear = self.empty_rx_buffer()
                if not status_clear:
                    rospy.logdebug(f'FAILED to send/read cmds after cleaning the rx buffer')
                    return False, []

        return True, rx


class Harrier10xController:
    def __init__(self, serial_number, lib_path):
        self.comm = HarrierCom(serial_number, lib_path)

    def start(self):
        self.comm.connect()
        self.comm.empty_rx_buffer()
        self.lens_homing()
        self.comm.send_and_read(Visca.focus_manual, Visca.cmd_rx_sz)
        self.comm.send_and_read(Visca.icr_night, Visca.cmd_rx_sz)
        self.comm.send_and_read(Visca.ae_iris_priority, Visca.cmd_rx_sz)

    def set_pos_zoom(self, val):
        status, _ = self.comm.send_and_read(Visca.zoom_direct(val), Visca.cmd_rx_sz, timeout=5000)
        return status

    def set_pos_focus(self, val):
        status, _ = self.comm.send_and_read(Visca.focus_direct(val), Visca.cmd_rx_sz)
        return status

    def set_pos_iris(self, val):
        status, _ = self.comm.send_and_read(Visca.iris_direct(val), Visca.cmd_rx_sz)
        return status

    def get_status_zfi(self):
        zoom, focus, iris = None, None, None

        status_z, reply = self.comm.send_and_read(Visca.zoom_inq, Visca.inq_zoom_rx_sz)
        if status_z:
            zoom = Visca.decode_zoom_inq(reply)

        status_f, reply = self.comm.send_and_read(Visca.focus_inq, Visca.inq_focus_rx_sz)
        if status_f:
            focus = Visca.decode_focus_inq(reply)

        status_i, reply = self.comm.send_and_read(Visca.iris_inq, Visca.inq_iris_rx_sz)
        if status_i:
            iris = Visca.decode_iris_inq(reply)

        return zoom, focus, iris

    def lens_homing(self):
        status, _ = self.comm.send_and_read(Visca.initialize_lens, Visca.cmd_rx_sz, timeout=5000, sleep=1)
        time.sleep(1)  # it seems there is reset procedure which takes time
        return status


if __name__ == '__main__':
    hc = Harrier10xController('1-1708099383-AS10LHD-2', '../HarrierUSBSDK-linux-1.0.0/libHarrierCommsUSB.so')
    z = 2000
    f = 0x1250
    i = 12
    print(hex(z), z)
    print(hex(f), f)
    print(hex(i), i)
    hc.set_pos_zoom(z)
    hc.set_pos_focus(f)
    hc.set_pos_iris(i)
    z, f, i = hc.get_status_zfi()
    print(hex(z), z)
    print(hex(f), f)
    print(hex(i), i)
