/*
Software for the Autonomous Robotic Observation and Behavioral Analysis system. 

Camera optics driver

Copyright 2025 Jiri Ulrich 

Commercial use of the software requires written consent of the copyright holders. 
For Open Research and Educational use, the following applies:

Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.



*/




/************************************************************************
 * Active Silicon Harrier SDK
 *
 * THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 * ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
 * PARTICULAR PURPOSE.
 *
 * Copyright (c) 2022 Active Silicon Limited. All rights reserved.
 ************************************************************************/

#ifndef HARRIER_COMMS_USB_H_
#define HARRIER_COMMS_USB_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdlib.h>

typedef int HarrierUSBHandle;
typedef void * HarrierCommsHandle;

typedef enum {
    HarrierCommsOK = 0,
    HarrierCommsAccessFail,
    HarrierCommsRXOutofBounds,
    HarrierCommsRXBufferOverflow,
    HarrierCommsRXEmpty,
    HarrierCommsOpenFail,
    HarrierCommsBadIndex,
    HarrierCommsBadHandle,
    HarrierCommsBufferTooSmall,
    HarrierCommsBadArgs,
} HarrierCommsError_t;

/**
 * @brief Open connection to the USB device using device index
 *      Takes in an index value for the USB device to open 
 *      to allow messages to be sent and received. 
 * @param handle    handle to the instance of the functions.
 * @param index     index of the USB device to open.
 * @return HarrierCommsError_t 
 */
HarrierCommsError_t HarrierCommsUSBOpenByIndex(int *handle, int index);

/**
 * @brief Close the connection and destroy handle/object 
 *  created by this library
 * 
 * @param handle 
 * @return HarrierCommsError_t 
 */
HarrierCommsError_t HarrierCommsUSBClose(int handle);

/**
 * @brief Transmit data to the USB device
 *      This take in a buffer with the data and size of message to send
 * 
 * @param handle 
 * @param buffer 
 * @param size
 * @return HarrierCommsError_t 
 */
HarrierCommsError_t HarrierCommsUSBTransmit(int handle, const void *msg, size_t size);

/**
 * @brief Harrier Receive bytes from device
 *      Takes in a buffer for the bytes, and number of bytes to read
 *      and returns the number of bytes read to the pointer. Will 
 *      continue to read until correct number of bytes acquired
 *      or until the timer runs out.
 *    
 * @param handle 
 * @param buffer    data buffer
 * @param bufsize   size of the data buffer that you want to read into
 * @param bytes     number of bytes to read and returns bytes read
 * @param timeout   timeout in milliseconds
 * @return HarrierCommsError_t  Returns error code
 */
HarrierCommsError_t HarrierCommsUSBReceive(int handle, void *buffer, size_t bufsize, unsigned char *bytes, int timeout); // Read until correct number of bytes or timeout;
// time mutex

#ifdef __cplusplus
}
#endif
#endif
