# Software for the Autonomous Robotic Observation and Behavioral Analysis system
#
# Agent high-level control
#
# Copyright 2025 Tomas Roucek 
#
# Commercial use of the software requires written consent of the copyright holders. 
#
# For Open Research and Educational use, the following applies:
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at

#   http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#!/usr/bin/env python

# Standard imports
import sys
import enum

# Third-party imports
import numpy as np
import cv2
import actionlib
import rospy

# ROS message imports
from sensor_msgs.msg import Image, JointState
from geometry_msgs.msg import Point, PoseStamped
import std_msgs.msg
from cv_bridge import CvBridge, CvBridgeError

# Project imports
from rr_interaction.msg import InteractionAction, InteractionGoal, InteractionResult, InteractionFeedback
from rr_msgs.msg import AgentControl, AgentStatus, BeePositionArray, ActuatorStatus, ActuatorDebug
from interaction.utils import InteractionUtils

# Enum for state machine states
class InteractionState(enum.Enum):
    READY = 0
    START = 1
    STATE_CHECK = 2
    PLAN_MOVEMENT = 3
    LOWER_AGENT = 4
    APPROACH = 5
    INTERACT = 6
    DEPART = 7

# State timeout durations in seconds
STATE_TIMEOUTS = {
    InteractionState.READY: float('inf'),  # No timeout for READY state
    InteractionState.START: 10.0,          # 10 seconds to detect queen and retract agent
    InteractionState.STATE_CHECK: 20.0,    # 15 seconds to verify queen stability
    InteractionState.PLAN_MOVEMENT: 20.0,  # 10 seconds to plan the movement
    InteractionState.LOWER_AGENT: 20.0,     # 20 seconds to lower agent
    InteractionState.APPROACH: 30.0,       # 30 seconds to approach queen
    InteractionState.INTERACT: float('inf'),   # No timeout for interaction state
    InteractionState.DEPART: 20.0          # 20 seconds to depart
}

class InteractionNode:
    def __init__(self):
        # Initialize the ROS node
        rospy.init_node('interaction_node')
        
        # Load parameters from the parameter server
        self._load_parameters()
        
        # Initialize utils and bridge
        self.utils = InteractionUtils(self._get_utils_config())
        self.bridge = CvBridge()
        
        # State machine variables
        self.current_state = InteractionState.READY
        self.last_state_change_time = rospy.Time.now()
        
        # Queen tracking variables
        self.queen_position = [0.0, 0.0, 0.0]  # [x, y, z]
        self.queen_position_history = []
        self.queen_pixel_position = [0, 0]
        self.camera_position = [0.0, 0.0]
        self.last_queen_detection = rospy.Time.now()
        self.queen_lost = True
        self.queen_resting = False
        self.queen_centered = False
        self.distance_to_queen = 0.0
        
        # Position and movement variables
        self.current_position_x = 0.0
        self.current_position_y = 0.0
        self.current_position_z = 0.0
        self.current_z_height = 0.0
        self.inpos_x = False
        self.inpos_y = False
        self.last_cmd_x = None
        self.last_cmd_y = None
        self.last_cmd_z = None
        self.last_cmd_max_speed = None
        self.last_z_position = None
        self.last_z_velocity = None
        
        # Agent variables
        self.agent_position = 0.0
        self.agent_current = 0.0
        self.agent_retracted = False
        self.agent_lowered = False
        self.current_fill_percentage = 0
        
        # Path planning variables
        self.heatmap = None
        self.heatmap_timestamp = rospy.Time(0)
        self.lower_position = None
        self.lower_position_uv = None
        self.lowering_metric_position = None
        self.lowering_xy_reached = False
        self.planned_path = []
        self.planned_path_uv = None
        self.planned_path_metric = []
        self.path_points = []
        self.current_path_index = 0
        self.path_planned = False
        
        # Status tracking
        self.queen_reached = False
        self.interaction_complete = False
        self.switch_left = False
        self.switch_right = False
        self.switch_top = False 
        self.switch_bottom = False
        
        # Action server variables
        self.action_goal = None
        self.action_result = InteractionResult()
        self.action_feedback = InteractionFeedback()
        self.action_start_time = None
        
        # Initialize publishers, subscribers, and timers
        self._init_pubs_subs()
        self._init_action_server()
        
        rospy.loginfo("Interaction node initialized in READY state")
    
    def _load_parameters(self):
        """Load parameters from the parameter server"""
        self.update_rate = rospy.get_param('~update_rate', 10.0)
        self.timeout = rospy.get_param('~timeout', 30.0)
        self.debug = rospy.get_param('~debug', False)
        self.max_force = rospy.get_param('~max_force', 0.01)
        self.image_width = rospy.get_param('~image_width', 1080)
        self.image_height = rospy.get_param('~image_height', 1920)
    
    def _get_utils_config(self):
        """Get configuration for InteractionUtils"""
        return {
            'min_queen_size': rospy.get_param('~min_queen_size', 150),
            'max_queen_size': rospy.get_param('~max_queen_size', 3500),
            'stable_position_threshold': rospy.get_param('~stable_position_threshold', 0.003),
            'stable_position_measurements': rospy.get_param('~stable_position_measurements', 20),
            'image_center_threshold': rospy.get_param('~image_center_threshold', 0.15),
            'queen_head_distance': rospy.get_param('~queen_head_distance', 0.05),
            'queen_head_angle_offset': rospy.get_param('~queen_head_angle_offset', -1.5),
            'path_search_distance': rospy.get_param('~path_search_distance', 5),
            'path_resolution': rospy.get_param('~path_resolution', 0.01),
            'heatmap_threshold': rospy.get_param('~heatmap_threshold', 10),
            'agent_retracted_position': rospy.get_param('~agent_retracted_position', 0.0),
            'agent_extended_position': rospy.get_param('~agent_extended_position', 0.1),
            'agent_position_tolerance': rospy.get_param('~agent_position_tolerance', 0.02)
        }
        
    def _init_pubs_subs(self):
        """Initialize publishers and subscribers"""
        # Publishers
        self.xy_cmd_pub = rospy.Publisher('xy_cmd', JointState, queue_size=1)
        self.z_cmd_pub = rospy.Publisher('z_axis/desired_joint_states', JointState, queue_size=1)
        self.agent_control_pub = rospy.Publisher('agent_control', AgentControl, queue_size=10)
        self.state_pub = rospy.Publisher('state', std_msgs.msg.String, queue_size=1)
        self.planned_path_pub = rospy.Publisher('planned_path', Image, queue_size=1)
        self.lower_position_pub = rospy.Publisher('lower_position', Point, queue_size=1)
        self.lowering_viz_pub = rospy.Publisher('lowering_position_viz', Image, queue_size=1)
        
        # Subscribers
        self.queen_position_sub = rospy.Subscriber('queen_position', BeePositionArray, self.queen_position_callback)
        self.z_status_sub = rospy.Subscriber('z_axis/joint_states', JointState, self.z_position_callback)
        self.agent_status_sub = rospy.Subscriber('agent_status', AgentStatus, self.agent_fill_status_callback)
        self.xy_position_sub = rospy.Subscriber('xy_position', PoseStamped, self.xy_position_callback)
        self.xy_status_sub = rospy.Subscriber('xy_status', ActuatorStatus, self.xy_status_callback)
        self.xy_status_debug_sub = rospy.Subscriber('xy_status_debug', ActuatorDebug, self.xy_status_debug_callback)
        self.heatmap_sub = rospy.Subscriber('bee_heatmap', Image, self.heatmap_callback)
        
        # Create a timer for periodic processing
        self.timer = rospy.Timer(rospy.Duration(1.0/self.update_rate), self.timer_callback)
    
    def _init_action_server(self):
        """Initialize the action server"""
        self.action_server = actionlib.SimpleActionServer('/interaction', InteractionAction, 
                                                         execute_cb=self.execute_action_cb, 
                                                         auto_start=False)
        self.action_server.start()
    
    # ------------------------------------------------------------------------------------
    #                             ROS Callback Methods
    # ------------------------------------------------------------------------------------
    
    def heatmap_callback(self, msg):
        """Process heatmap image updates."""
        try:
            self.heatmap = self.bridge.imgmsg_to_cv2(msg, "bgr8")
            self.heatmap_timestamp = rospy.Time.now()
        except CvBridgeError as e:
            rospy.logerr(f"Error converting image: {e}")
    
    def z_position_callback(self, msg):
        """Process Z axis position updates."""
        if len(msg.position) > 0:
            self.agent_position = msg.position[0]
            self.agent_retracted = self.utils.check_agent_position(
                self.agent_position, self.utils.agent_retracted_position)
    
    def agent_fill_status_callback(self, msg):
        """Process agent fill status updates."""
        self.current_fill_percentage = msg.filled_percentage
    
    def xy_position_callback(self, msg):
        """Process XY position updates."""
        self.current_position_x = msg.pose.position.x
        self.current_position_y = msg.pose.position.y
        self.current_position_z = msg.pose.position.z
    
    def xy_status_callback(self, msg):
        """Process status updates from the xy table actuator."""
        # Track switch states
        self.switch_left = msg.switchLeft
        self.switch_right = msg.switchRight 
        self.switch_top = msg.switchTop
        self.switch_bottom = msg.switchBottom

        # Check if any switches are triggered
        switches_triggered = any([
            self.switch_left,
            self.switch_right, 
            self.switch_top,
            self.switch_bottom
        ])

        # If we hit side of the table lets depart and depart the agent form queen
        if switches_triggered and self.current_state not in [InteractionState.READY, InteractionState.START]:
            rospy.logwarn(
                "End switch triggered at x=%.3f, y=%.3f! Left: %s Right: %s Top: %s Bottom: %s. Initiating docking procedure",
                self.current_position_x,
                self.current_position_y, 
                self.switch_left,
                self.switch_right,
                self.switch_top, 
                self.switch_bottom
            )
            
            self.current_state = InteractionState.DEPART
        
        # Publish current state
        state_msg = std_msgs.msg.String()
        state_msg.data = self.current_state.name
        self.state_pub.publish(state_msg)

    def xy_status_debug_callback(self, msg):
        """Process debug status updates from the xy table actuator."""
        self.inpos_x = msg.inpos_x
        self.inpos_y = msg.inpos_y
    
    def queen_position_callback(self, msg):
        """Process queen position data."""
        current_time = rospy.Time.now()
        # wait for 4 positions to record
        if len(msg.positions) >= 4:  
            # Check if queen's size is within expected range
            if msg.positions[0].size >= self.utils.min_queen_size and msg.positions[0].size <= self.utils.max_queen_size:
                # Store queen's metric coordinates
                self.queen_position = [
                    msg.positions[1].x,  # x in hive coordinates
                    msg.positions[1].y,  # y in hive coordinates
                    0.0  # z coordinate
                ]
                
                # Store camera position
                self.camera_position = [
                    msg.positions[2].x,
                    msg.positions[2].y
                ]
                
                # Store pixel coordinates for heatmap operations
                self.queen_pixel_position = [
                    msg.positions[0].u,
                    msg.positions[0].v
                ]
                
                # Check if queen is centered in image
                self.queen_centered = self.utils.check_queen_centered(
                    msg.positions[0].u, 
                    msg.positions[0].v,
                    self.image_width, self.image_height)
                
                # Calculate distance to queen
                self.distance_to_queen = np.sqrt(
                    (self.queen_position[0] - self.camera_position[0])**2 + 
                    (self.queen_position[1] - self.camera_position[1])**2
                )
                
                self.queen_lost = False
                self.last_queen_detection = current_time
                
                # Update position history for stability checking
                self.queen_position_history.append(self.queen_position[:2])
                if len(self.queen_position_history) > self.utils.stable_position_measurements:
                    self.queen_position_history.pop(0)
                
                # Check if queen is stable
                self.queen_resting = self.utils.check_queen_stability(self.queen_position_history)
                
                if self.debug:
                    rospy.logdebug(f"Queen detected at ({self.queen_position[0]:.4f}, {self.queen_position[1]:.4f})")
                    rospy.logdebug(f"Queen centered: {self.queen_centered}, resting: {self.queen_resting}")
            
        # Mark queen as lost if not seen for 3+ seconds
        if (current_time - self.last_queen_detection).to_sec() > 3.0:
            self.queen_lost = True
            self.queen_resting = False
            self.queen_centered = False
            
            if (current_time - self.last_queen_detection).to_sec() > 5.0:
                self.queen_position_history = []
    
    def timer_callback(self, event):
        """Process state machine logic periodically."""
        if not self.action_server.is_active() and self.current_state != InteractionState.READY:
            # Return to READY if action server not active
            self.current_state = InteractionState.READY
            return
            
        if self.current_state == InteractionState.READY:
            return  # Just wait in READY state
        elif self.current_state == InteractionState.START:
            self.execute_start_state()
        elif self.current_state == InteractionState.STATE_CHECK:
            self.execute_state_check()
        elif self.current_state == InteractionState.PLAN_MOVEMENT:
            self.execute_plan_movement()
        elif self.current_state == InteractionState.LOWER_AGENT:
            self.execute_lower_agent()
        elif self.current_state == InteractionState.APPROACH:
            self.execute_approach()
        elif self.current_state == InteractionState.INTERACT:
            self.execute_interact()
        elif self.current_state == InteractionState.DEPART:
            self.execute_depart()
    
    def execute_action_cb(self, goal):
        """Handle action server execution."""
        rospy.loginfo("Received interaction action goal")
        self.action_goal = goal
        self.action_start_time = rospy.Time.now()
        
        # Transition to START state and reset flags
        self.current_state = InteractionState.START
        self.last_state_change_time = rospy.Time.now()
        
        self.queen_lost = True
        self.queen_resting = False
        self.path_planned = False
        self.agent_lowered = False
        self.queen_reached = False
        self.interaction_complete = False
        
        while not rospy.is_shutdown():
            if self.action_server.is_preempt_requested():
                rospy.loginfo("Interaction action preempted")
                self.action_server.set_preempted()
                return
            
            if self.current_state == InteractionState.START and self.interaction_complete:
                self.action_result.success = True
                elapsed_time = (rospy.Time.now() - self.action_start_time).to_sec()
                self.action_result.actual_interaction_duration = elapsed_time
                self.action_result.message = "Interaction completed successfully"
                self.action_server.set_succeeded(self.action_result)
                self.current_state = InteractionState.READY
                return
            
            # Check for state timeout
            time_in_current_state = (rospy.Time.now() - self.last_state_change_time).to_sec()
            state_timeout = STATE_TIMEOUTS[self.current_state]
            
            if time_in_current_state > state_timeout and state_timeout != float('inf'):
                rospy.logwarn(f"State {self.current_state.name} timed out after {time_in_current_state:.1f} seconds")
                self.action_result.success = False
                self.action_result.actual_interaction_duration = (rospy.Time.now() - self.action_start_time).to_sec()
                self.action_result.message = f"State {self.current_state.name} timed out"
                self.action_server.set_aborted(self.action_result)
                self.current_state = InteractionState.READY
                return
            
            # Update and publish feedback
            self.action_feedback.current_state = self.current_state.name
            self.action_feedback.elapsed_time = (rospy.Time.now() - self.action_start_time).to_sec()
            self.action_feedback.distance_to_queen = self.distance_to_queen
            self.action_feedback.queen_detected = not self.queen_lost
            self.action_feedback.queen_resting = self.queen_resting
            self.action_feedback.interaction_progress = 0.0  
            self.action_server.publish_feedback(self.action_feedback)
            
            rospy.sleep(0.1)
    
    def execute_start_state(self):
        """Execute logic for the START state"""
        if not self.agent_retracted:
            rospy.loginfo("Retracting agent Z-axis...")
            self.send_z_command(self.utils.agent_retracted_position)
            return
        
        if self.queen_lost:
            rospy.loginfo("Waiting for queen detection...")
            return
        
        rospy.loginfo("Start state conditions met: agent retracted, queen detected")
        self.transition_to_state(InteractionState.STATE_CHECK)
    
    def execute_state_check(self):
        """Execute logic for the STATE_CHECK state"""
        if self.queen_lost:
            rospy.logwarn("Lost sight of queen during STATE_CHECK, returning to START")
            self.transition_to_state(InteractionState.START)
            return
        
        if len(self.queen_position_history) < self.utils.stable_position_measurements:
            remaining = self.utils.stable_position_measurements - len(self.queen_position_history)
            rospy.loginfo(f"Collecting queen position measurements... {remaining} more needed")
            return
        
        if not self.queen_resting:
            rospy.loginfo("Waiting for queen to stop moving...")
            return
        
        rospy.loginfo(f"Queen resting at position ({self.queen_position[0]:.4f}, {self.queen_position[1]:.4f}, u: {self.queen_pixel_position[0]}, v: {self.queen_pixel_position[1]})")
        self.transition_to_state(InteractionState.PLAN_MOVEMENT)
    
    def execute_plan_movement(self):
        """Execute logic for the PLAN_MOVEMENT state"""
        if self.heatmap is None or (rospy.Time.now() - self.heatmap_timestamp).to_sec() > 2.0:
            rospy.loginfo("Waiting for recent heatmap...")
            return
        
        if self.queen_lost or not self.queen_resting:
            rospy.logwarn("Lost sight of queen or queen started moving, returning to START")
            self.transition_to_state(InteractionState.START)
            return
        
        # Get lowering position and visualization in UV coordinates
        position_result_uv, lowering_viz = self.utils.find_lowering_position(
            self.heatmap,
            self.queen_pixel_position,
            self.queen_angle if hasattr(self, 'queen_angle') else None
        )
        
        if not position_result_uv:
            rospy.logwarn("Could not find suitable lowering position, retrying...")
            return
        
        # Convert UV coordinates to metric for robot control
        self.lower_position_uv = position_result_uv
        
        PIXEL_TO_METER_SCALE = -0.00007    # This should be calibrated for your setup
        
        # Calculate relative offset from image center in pixels
        du = position_result_uv[0] - self.image_width/2
        dv = position_result_uv[1] - self.image_height/2
        rospy.loginfo(f"Lowering position: u: {position_result_uv[0]}, v: {position_result_uv[1]}") 
        #du dv: 
        rospy.loginfo(f"du: {du}, dv: {dv}")
        # Convert pixel offsets to metric offsets
        dx = du * PIXEL_TO_METER_SCALE
        dy = dv * PIXEL_TO_METER_SCALE
        
        # Calculate absolute metric position
        self.lowering_metric_position = [
            self.camera_position[0] + dx,
            self.camera_position[1] + dy,
            position_result_uv[2]
        ]
        
        # Publish visualization if available
        if lowering_viz is not None:
            try:
                viz_msg = self.bridge.cv2_to_imgmsg(lowering_viz, "bgr8")
                self.lowering_viz_pub.publish(viz_msg)
            except CvBridgeError as e:
                rospy.logerr(f"Error converting lowering visualization: {e}")
        
        # Plan path in UV coordinates
        self.planned_path_uv = self.utils.plan_path_to_queen(
            self.heatmap,
            self.lower_position_uv,
            self.queen_pixel_position,
            self.queen_position[2]
        )
        
        if not self.planned_path_uv:
            rospy.logwarn("Could not plan path to queen, retrying...")
            return
        
        # Convert planned path to metric coordinates
        self.planned_path_metric = []
        for uv_point in self.planned_path_uv:
            # Calculate relative offset from image center in pixels
            du = uv_point[0] - self.image_width/2
            dv = uv_point[1] - self.image_height/2
            
            # Convert pixel offsets to metric offsets
            dx = du * PIXEL_TO_METER_SCALE
            dy = dv * PIXEL_TO_METER_SCALE
            
            # Add offsets to camera position
            metric_point = [
                self.camera_position[0] + dx,
                self.camera_position[1] + dy,
                uv_point[2]
            ]
            self.planned_path_metric.append(metric_point)
        
        # Reset the current path index
        self.current_path_index = 0
        
        # Publish the metric position for visualization
        lower_point = Point()
        lower_point.x = self.lowering_metric_position[0]
        lower_point.y = self.lowering_metric_position[1]
        lower_point.z = self.lowering_metric_position[2]
        self.lower_position_pub.publish(lower_point)
        
        # Visualize the path
        viz_img = self.utils.visualize_planned_path(
            self.heatmap,
            self.queen_pixel_position,
            self.lower_position_uv,
            self.planned_path_uv
        )
        
        if viz_img is not None:
            try:
                msg = self.bridge.cv2_to_imgmsg(viz_img, "bgr8")
                self.planned_path_pub.publish(msg)
            except CvBridgeError as e:
                rospy.logerr(f"Error converting path visualization: {e}")
        
        self.path_planned = True
        rospy.loginfo(f"Movement planned: Lowering at ({self.lowering_metric_position[0]:.4f}, {self.lowering_metric_position[1]:.4f})")
        rospy.loginfo(f"Path planned with {len(self.planned_path_metric)} points")
        self.transition_to_state(InteractionState.LOWER_AGENT)
    
    def execute_lower_agent(self):
        """Execute logic for the LOWER_AGENT state"""
        if self.lowering_metric_position is None:
            rospy.logwarn("No valid lowering position. Returning to PLAN_MOVEMENT.")
            self.transition_to_state(InteractionState.PLAN_MOVEMENT)
            return
        
        if self.queen_lost or not self.queen_resting:
            rospy.logwarn("Lost sight of queen or queen started moving during LOWER_AGENT, returning to START")
            self.transition_to_state(InteractionState.START)
            return
        
        if not hasattr(self, 'lowering_xy_reached') or not self.lowering_xy_reached:
            # Use metric coordinates for robot control
            self.goto_absolute(
                self.lowering_metric_position[0], 
                self.lowering_metric_position[1], 
                self.utils.agent_retracted_position
            )
            
            if self.position_reached(self.lowering_metric_position[0], self.lowering_metric_position[1]):
                rospy.loginfo(f"Reached lowering position: ({self.lowering_metric_position[0]:.4f}, {self.lowering_metric_position[1]:.4f})")
                self.lowering_xy_reached = True
            return
        
        if not self.agent_lowered:
            target_z = self.queen_position[2] + 1  # Z coordinate is already in metric
            # First lower the Z-axis
            self.send_z_command(target_z, velocity=0.001)
            self.current_z_height = target_z
            if self.utils.check_agent_position(self.agent_position, target_z):
                # Once Z-axis is in position, start the agent
                self.agent_lowered = True
                rospy.loginfo(f"Z-axis lowered to {target_z:.4f}, starting agent")
        
        if self.lowering_xy_reached and self.agent_lowered:
            rospy.loginfo("Agent successfully lowered. Moving to APPROACH state.")
            self.transition_to_state(InteractionState.APPROACH)
    
    def execute_approach(self):
        """Execute logic for the APPROACH state"""
        if self.queen_lost or not self.queen_resting:
            rospy.logwarn("Lost sight of queen or queen started moving during APPROACH, returning to START")
            self.transition_to_state(InteractionState.START)
            return
        
        if len(self.planned_path_metric) == 0 or self.current_path_index >= len(self.planned_path_metric):
            rospy.logwarn("No valid path available. Returning to PLAN_MOVEMENT.")
            self.transition_to_state(InteractionState.PLAN_MOVEMENT)
            return
        
        # Get the current target point from the path (in metric coordinates)
        target_point = self.planned_path_metric[self.current_path_index]
        
        # Move to the target point using metric coordinates
        self.goto_absolute(
            target_point[0], target_point[1], self.current_z_height, max_speed=0.001
        )
        
        # Check if we reached the target point
        if self.position_reached(target_point[0], target_point[1], self.current_z_height, tolerance=0.001):
            rospy.loginfo(f"Reached path point {self.current_path_index + 1}/{len(self.planned_path_metric)}: ({target_point[0]:.4f}, {target_point[1]:.4f})")
            self.current_path_index += 1
            
            # Check if we've reached the end of the path
            if self.current_path_index >= len(self.planned_path_metric):
                rospy.loginfo("Reached the queen's interaction position.")
                self.queen_reached = True
                self.transition_to_state(InteractionState.INTERACT)
    
    def execute_interact(self):
        """Execute logic for the INTERACT state"""
        if self.queen_lost or not self.queen_resting:
            rospy.logwarn("Lost sight of queen or queen started moving during INTERACT, returning to START")
            self.transition_to_state(InteractionState.START)
            return
        
        # Check if we have interaction_start_time, if not, initialize it
        if not hasattr(self, 'interaction_start_time'):
            self.interaction_start_time = rospy.Time.now()
            # Initialize interaction mechanism state
            self.interaction_active = False
            
            # Get duration from action goal if available, otherwise use parameter
            if hasattr(self.action_goal, 'interaction_duration') and self.action_goal.interaction_duration > 0:
                self.interaction_duration = self.action_goal.interaction_duration
                rospy.loginfo(f"Using interaction duration from action goal: {self.interaction_duration} seconds")
            else:
                self.interaction_duration = rospy.get_param('~interaction_duration', 5.0)  # Default 5 seconds
                rospy.loginfo(f"Using interaction duration from parameter: {self.interaction_duration} seconds")
            
            # Get percentage from action goal if available, otherwise use parameter
            if hasattr(self.action_goal, 'interaction_percentage') and self.action_goal.interaction_percentage > 0:
                self.interaction_percentage = self.action_goal.interaction_percentage
                rospy.loginfo(f"Using interaction percentage from action goal: {self.interaction_percentage}%")
            else:
                self.interaction_percentage = rospy.get_param('~interaction_percentage', 75)  # Default 75%
                rospy.loginfo(f"Using interaction percentage from parameter: {self.interaction_percentage}%")
            
            self.interaction_agent_activated = False
            rospy.loginfo(f"Starting interaction sequence for {self.interaction_duration} seconds at {self.interaction_percentage}% control")
        
        # Activate the interaction agent if not already activated
        if not self.interaction_agent_activated:
            # Send command to activate the interaction agent using percentage control
            agent_msg = AgentControl()
            agent_msg.percentage = self.interaction_percentage
            agent_msg.flow_rate = -1.0  # Use percentage mode
            self.agent_control_pub.publish(agent_msg)
            
            self.interaction_agent_activated = True
            rospy.loginfo(f"Interaction agent activated with {self.interaction_percentage}% control")
            self.interaction_active = True
        
        # Check if we've been interaction for the required duration
        elapsed_time = (rospy.Time.now() - self.interaction_start_time).to_sec()
        
        # Provide feedback about interaction progress
        if elapsed_time < self.interaction_duration:
            if int(elapsed_time) % 1 == 0:  # Log every second
                rospy.loginfo(f"Interaction in progress: {elapsed_time:.1f}/{self.interaction_duration} seconds, fill level: {self.current_fill_percentage}%")
            self.action_feedback.interaction_progress = elapsed_time / self.interaction_duration
            self.action_server.publish_feedback(self.action_feedback)
            return
        
        # Deactivate the interaction agent (set percentage to 0)
        agent_msg = AgentControl()
        agent_msg.percentage = 0
        agent_msg.flow_rate = -1.0  # Use percentage mode
        self.agent_control_pub.publish(agent_msg)
        
        rospy.loginfo(f"Interaction complete after {elapsed_time:.2f} seconds")
        
        # Clean up the state variables
        self.interaction_agent_activated = False
        self.interaction_active = False
        delattr(self, 'interaction_start_time')
        
        # Mark interaction as complete and transition to DEPART
        self.interaction_complete = True
        self.transition_to_state(InteractionState.DEPART)
    
    def execute_depart(self):
        """Execute logic for the DEPART state"""
        # Track the depart state progress
        if not hasattr(self, 'depart_state'):
            self.depart_state = 0
            rospy.loginfo("Starting departure sequence")
        
        # State 0: Stop agent and retract Z-axis
        if self.depart_state == 0:
            # First stop the agent
            self.send_agent_command(percentage=0)
            # Then retract the Z-axis
            self.send_z_command(self.utils.agent_retracted_position)
            
            if self.agent_retracted:
                rospy.loginfo("Agent Z-axis retracted for departure")
                self.depart_state = 1
                self.interaction_complete = True
                rospy.loginfo("Departure complete, interaction sequence finished")
                self.transition_to_state(InteractionState.READY)
            return

    def position_reached(self, target_x, target_y, target_z=None, tolerance=0.002):
        """Check if the target position has been reached."""
        return self.utils.check_position_reached(
            self.current_position_x, 
            self.current_position_y,
            target_x, 
            target_y,
            self.inpos_x, 
            self.inpos_y,
            self.agent_position if target_z is not None else None,
            target_z,
            tolerance
        )

    def send_agent_command(self, percentage=0, flow_rate=-1.0):
        """Send a command to control the agent."""
        msg = AgentControl()
        msg.percentage = percentage
        msg.flow_rate = flow_rate
        self.agent_control_pub.publish(msg)
        
        if flow_rate >= 0:
            rospy.loginfo(f"Setting agent flow rate to {flow_rate} ml/s")
        else:
            rospy.loginfo(f"Setting agent interacton intensity to {percentage}%")

    def send_z_command(self, z_position, velocity=0.01):
        """Send a command to move the Z-axis."""
        # Skip if same command as last time
        if (self.last_z_position == z_position and 
            self.last_z_velocity == velocity):
            #return
            pass

        # Store current command
        self.last_z_position = z_position
        self.last_z_velocity = velocity
        
        cmd = JointState()
        cmd.name = ["z_axis_1"]
        cmd.position = [z_position]
        cmd.velocity = [velocity]
        cmd.effort = [0.0]
        self.z_cmd_pub.publish(cmd)
        rospy.loginfo(f"Moving Z-axis to position: {z_position:.4f}")

    def transition_to_state(self, new_state):
        """Transition to a new state"""
        old_state = self.current_state
        self.current_state = new_state
        self.last_state_change_time = rospy.Time.now()  # Update state change time
        rospy.loginfo(f"State transition: {old_state.name} -> {new_state.name}")
    
    def goto_relative(self, rx, ry, rz=0.0, max_speed_x=0.005, max_speed_y=0.005, max_speed_z=0.005):
        """Move relative to current position."""
        cmd = JointState()
        cmd.name = ["x_axis", "y_axis"]
        cmd.position = [self.current_position_x + rx, self.current_position_y + ry]
        cmd.velocity = [max_speed_x, max_speed_y]
        cmd.effort = [self.max_force, self.max_force]
        
        self.xy_cmd_pub.publish(cmd)
        
        if rz != 0:
            z_cmd = JointState()
            z_cmd.name = ["z_axis_1"]
            z_cmd.position = [self.current_position_z + rz]
            z_cmd.velocity = [max_speed_z]
            self.z_cmd_pub.publish(z_cmd)
        
        rospy.loginfo(f"Moving relative: dx={rx:.4f}, dy={ry:.4f}, dz={rz:.4f}")

    def goto_absolute(self, x, y, z=0, max_speed=0.01, over_limits=False):
        """Move to an absolute position."""
        # Skip if same command as last time
        if (self.last_cmd_x == x and 
            self.last_cmd_y == y and 
            self.last_cmd_z == z and 
            self.last_cmd_max_speed == max_speed):
            return
        
        # Store current command
        self.last_cmd_x = x
        self.last_cmd_y = y
        self.last_cmd_z = z
        self.last_cmd_max_speed = max_speed
        
        # Send XY command
        cmd = JointState()
        cmd.name = ["x_axis", "y_axis"]
        cmd.position = [x, y]
        cmd.velocity = [max_speed, max_speed]
        cmd.effort = [self.max_force, self.max_force]
        self.xy_cmd_pub.publish(cmd)
        
        # Send Z command if needed
        if z is not None:
            z_cmd = JointState()
            z_cmd.name = ["z_axis_1"]
            z_cmd.position = [z]
            z_cmd.velocity = [max_speed]
            self.z_cmd_pub.publish(z_cmd)
        
        rospy.loginfo(f"Moving to absolute: ({x:.4f}, {y:.4f}, {z if z is not None else 'unchanged'})")

    def run(self):
        """Run the node."""
        rospy.spin()

if __name__ == '__main__':
    try:
        node = InteractionNode()
        node.run()
    except rospy.ROSInterruptException:
        pass 
