# Software for the Autonomous Robotic Observation and Behavioral Analysis system
#
# Agent high-level control
#
# Copyright 2025 Tomas Roucek 
#
# Commercial use of the software requires written consent of the copyright holders. 
#
# For Open Research and Educational use, the following applies:
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at

#   http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#!/usr/bin/env python

import rospy
import actionlib
from rr_interaction.msg import InteractionAction, InteractionGoal
import sys

def interaction_client():
    """
    Example client for the interaction action server with configurable duration and percentage
    """
    # Initialize the ROS node
    rospy.init_node('interaction_client')
    
    # Create an action client
    client = actionlib.SimpleActionClient('/interaction', InteractionAction)
    
    # Wait for the server to start
    rospy.loginfo("Waiting for interaction action server...")
    client.wait_for_server()
    
    # Create a goal
    goal = InteractionGoal()
    
    # Set target queen ID
    goal.target_queen_id = "queen1"
    
    # Get command line arguments or use defaults
    try:
        # Get interaction duration from command line or use default
        if len(sys.argv) > 1:
            goal.interaction_duration = float(sys.argv[1])
        else:
            goal.interaction_duration = 10.0  # default 10 seconds
            
        # Get interaction percentage from command line or use default
        if len(sys.argv) > 2:
            goal.interaction_percentage = float(sys.argv[2])
        else:
            goal.interaction_percentage = 75.0  # default 75%
    except ValueError:
        rospy.logerr("Invalid arguments. Usage: interaction_client_example.py [duration] [percentage]")
        return
    
    rospy.loginfo(f"Sending interaction goal: duration={goal.interaction_duration}s, percentage={goal.interaction_percentage}%")
    
    # Send the goal
    client.send_goal(
        goal,
        feedback_cb=feedback_callback
    )
    
    # Wait for the result
    client.wait_for_result()
    
    # Get the result
    result = client.get_result()
    
    if result.success:
        rospy.loginfo(f"Interaction successful! Duration: {result.actual_interaction_duration}s")
    else:
        rospy.logwarn(f"Interaction failed: {result.message}")
    
    return result

def feedback_callback(feedback):
    """Process feedback from the action server"""
    if feedback.queen_detected:
        status = "queen detected"
        if feedback.queen_resting:
            status += ", resting"
    else:
        status = "no queen"
    
    rospy.loginfo(f"State: {feedback.current_state}, Status: {status}, " +
                 f"Elapsed: {feedback.elapsed_time:.1f}s, " +
                 f"Progress: {feedback.interaction_progress:.1%}")

if __name__ == '__main__':
    try:
        result = interaction_client()
    except rospy.ROSInterruptException:
        rospy.loginfo("Program interrupted before completion") 
