# RR Interaction 

This ROS1 package **rr_approach** provides an example for interacting agent control

## Dependencies

- ROS Noetic
- rr_msgs: Contains message definitions used by this package

## State Machine

The interaction behavior is implemented as a state machine with the following states:

1. **START**: Initializes the interaction process and performs necessary setup.
   - Ensures the agent is retracted
   - Checks that the queen bee is detected
   - Verifies the queen is roughly centered in the image

2. **STATE_CHECK**: Detects the queen bee and checks if she is resting.
   - Takes multiple measurements of the queen's position (default: 20)
   - Establishes Z distance to the queen
   - Confirms the queen has moved less than 3mm in recent measurements
   - Only proceeds when the queen is stationary

3. **PLAN_MOVEMENT**: Assesses where to lower the agent and plans a path to the queen.
   - Uses a heatmap image to identify positions of other bees
   - Finds a clear spot (no bees) to lower the agent, preferably above the queen in X coordinates
   - Plans a path from the lowering point to the queen's head (approximately 3cm from center)
   - Visualizes the planned path for monitoring

4. **LOWER_AGENT**: Lowers the agent to a predefined position.
   - Moves the end effector to the planned lowering position
   - Lowers the agent to the same level as the queen to avoid crashing into the comb
   - Uses position control similar to xy_operator.cpp

5. **APPROACH**: Follows the planned path to approach the queen bee.
   - Follows the pre-planned path to the queen's head
   - Monitors for unexpected obstacles or queen movement
   - Adjusts approach as needed

6. **INTERACT**: Activates the agent to interact with the queen until a trigger condition is met.
   - Activates the agent interaction actuators using configurable interaction intensity control
   - Monitors interaction progress and fill level
   - Continues interaction for the specified duration
   - Deactivates the agent interaction when complete

7. **DEPART**: Safely moves the agent away from the queen after interaction.
   - Retracts the agent
   - Moves away from the queen
   - Returns to a safe position

## Implementation Details

### Queen Detection
- Uses size-based filtering to identify the queen bee
- Tracks queen position over time to establish stability
- Calculates average Z distance for proper approach

### Path Planning
- Uses heatmap analysis to find clear areas for agent lowering
- Prioritizes positions above the queen (in the direction of her head)
- Plans direct paths with obstacle awareness
- Provides visualization of planned paths

### Movement Control
- Implements absolute and relative movement functions
- Monitors position to determine when targets are reached
- Uses velocity control for smooth movement

### Agent Control
- Controls agent extension and retraction
- Uses percentage-based control for interaction intensity 
- Monitors agent position and interaction level through feedback
- Ensures safe operation with position tolerance checks

## Configuration Parameters

Key parameters that can be adjusted in the config file:

### Queen Detection
- `min_queen_size`: Minimum size for queen detection (default: 150)
- `max_queen_size`: Maximum size for queen detection (default: 3500)
- `stable_position_threshold`: Movement threshold for stability (default: 0.003m)
- `stable_position_measurements`: Number of measurements for stability check (default: 20)

### Path Planning
- `queen_head_distance`: Distance from queen center to head (default: 0.03m)
- `path_search_distance`: Maximum search radius for lowering position (default: 0.10m)
- `path_resolution`: Step size for path planning (default: 0.01m)
- `heatmap_threshold`: Threshold for considering a point free of bees (default: 10)

### Agent Parameters
- `agent_retracted_position`: Z position when agent is retracted (default: 0.0)
- `agent_extended_position`: Z position when agent is extended (default: 0.1)
- `agent_position_tolerance`: Tolerance for position checks (default: 0.005)

### Interaction Parameters
- `interaction_duration`: Duration of interaction in seconds (default: 5.0)
- `interaction_percentage`: Percentage value for interaction intensity control (default: 75)

## Action Server Interface

The interaction process is implemented as an Action Server, allowing clients to trigger and monitor the interaction operation. The action provides:

- **Goal**: 
  - `target_queen_id`: ID of the queen bee to interacted with 
  - `interaction_duration`: Maximum duration of interaction in seconds
  - `interaction_percentage`: Percentage value for interaction intensity (0-100)

- **Feedback**: 
  - Current state
  - Elapsed time
  - Distance to queen
  - Queen detection status
  - Queen resting status
  - Interaction progress (0.0 to 1.0)

- **Result**: 
  - Success status
  - Actual interaction duration
  - Status message

## Usage Example

```python
# Create a client
client = actionlib.SimpleActionClient('interaction', InteractionAction)

# Create and send goal
goal = InteractionGoal()
goal.target_queen_id = "queen1"
goal.interaction_duration = 10.0  # 10 seconds
goal.interaction_percentage = 75.0  # 75% control
client.send_goal(goal)

# Wait for result
client.wait_for_result()
result = client.get_result()
```

A complete example client is provided in `scripts/interaction_client_example.py`.

## Structure

- `scripts/`: Python executable scripts
- `src/`: Python module files
- `launch/`: ROS launch files
- `config/`: Configuration files 
